/*
* This file is part of Qccrypt, an open-source cross-platform frontend for ccrypt
* encryption tool.
* Copyright (C) 2006-2022  Philippe Beaureilles
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
* Contact e-mail: Philippe Beaureilles <qccrypt@free.fr>
* Program URL   : http://qccrypt.free.fr/
*
*/

#include <QtGui>
#include <QMessageBox>

#include "passworddialog.h"
#include "ui_passworddialog.h"
#include "passwordvalidator.h"

PasswordDialog::PasswordDialog(bool bUseSecondPassword, QString sessionPassword, QFileInfo sessionKeyFile, QWidget *parent) :
    QDialog(parent),
    pwdlg_ui(new Ui::PasswordDialog)
{
    pwdlg_ui->setupUi(this);

    bUseSecondPwd = bUseSecondPassword;
    currentKeyFileInfo = sessionKeyFile;

    // validators
    PasswordValidator* validator = new PasswordValidator(this);
    pwdlg_ui->firstPasswordEdit->setValidator(validator);
    if (bUseSecondPwd)
        pwdlg_ui->secondPasswordEdit->setValidator(validator);
    else
        pwdlg_ui->secondPasswordEdit->setEnabled(false);

    // communication
    connect(pwdlg_ui->okButton, SIGNAL(clicked()), SLOT(accept()));
    connect(pwdlg_ui->cancelButton, SIGNAL(clicked()), SLOT(reject()));
    connect(pwdlg_ui->browseKeyFileButton, SIGNAL(clicked()), SLOT(selectKeyFile()));

    connect(pwdlg_ui->enterThePasswordRadioButton, SIGNAL(clicked()), this, SLOT(enterThePasswordRadioButtonClicked()));
    connect(pwdlg_ui->useKeyFileRadioButton, SIGNAL(clicked()), this, SLOT(useKeyFileRadioButtonClicked()));

    connect(pwdlg_ui->firstPasswordEdit, SIGNAL(textChanged(const QString&)), SLOT(checkOkEnabled()));
    connect(pwdlg_ui->secondPasswordEdit, SIGNAL(textChanged(const QString&)), SLOT(checkOkEnabled()));

    connect(pwdlg_ui->firstPasswordEdit, SIGNAL(gotFocus()), SLOT(grabFirstPassword()));
    connect(pwdlg_ui->firstPasswordEdit, SIGNAL(lostFocus()), SLOT(release()));
    connect(pwdlg_ui->secondPasswordEdit, SIGNAL(gotFocus()), SLOT(grabSecondPassword()));
    connect(pwdlg_ui->secondPasswordEdit, SIGNAL(lostFocus()), SLOT(release()));

    connect(pwdlg_ui->showPasswordCheckBox, SIGNAL(clicked()), this, SLOT(showPasswordCheckBoxClicked()));
    connect(pwdlg_ui->rememberPasswordCheckBox, SIGNAL(clicked()), this, SLOT(rememberPasswordCheckBoxClicked()));
    connect(pwdlg_ui->useTemporaryFileCheckBox, SIGNAL(clicked()), this, SLOT(useTemporaryFileCheckBoxClicked()));
    connect(pwdlg_ui->rememberKeyFileCheckBox, SIGNAL(clicked()), this, SLOT(rememberKeyFileCheckBoxClicked()));

    //Read application's settings
    QString currentSetting;
    //  -- "Show password" setting --
    currentSetting = "Interface/ShowPassword";
    if (settings.contains(currentSetting))
    {
        pwdlg_ui->showPasswordCheckBox->setChecked(settings.value(currentSetting).toBool());
        showPasswordCheckBoxClicked();
    }
    else
        //By default, the "Show password" check box is not checked
        settings.setValue(currentSetting, false);
    //  -- "Remember password" setting --
    currentSetting = "Interface/RememberPassword";
    if (settings.contains(currentSetting))
    {
        if (settings.value(currentSetting).toBool())
        {
            pwdlg_ui->rememberPasswordCheckBox->setChecked(true);
            pwdlg_ui->firstPasswordEdit->setText(sessionPassword);
            if (bUseSecondPwd)
                pwdlg_ui->secondPasswordEdit->setText(sessionPassword);
        }
        else
            pwdlg_ui->rememberPasswordCheckBox->setChecked(false);
    }
    else
        //By default, the "Remember password" check box is not checked
        settings.setValue(currentSetting, false);

    //  -- "Use Temporary File" setting --
    currentSetting = "Interface/UseTemporaryFile";
    if (settings.contains(currentSetting))
    {
        if (settings.value(currentSetting).toBool())
        {
            pwdlg_ui->useTemporaryFileCheckBox->setChecked(true);
        }
        else
            pwdlg_ui->useTemporaryFileCheckBox->setChecked(false);
    }
    else
        //By default, the "Use Temporary File" check box is not checked
        settings.setValue(currentSetting, false);


    //  -- "Remember key file" setting --
    currentSetting = "Interface/RememberKeyFile";
    if (settings.contains(currentSetting))
    {
        if (settings.value(currentSetting).toBool())
        {
            pwdlg_ui->rememberKeyFileCheckBox->setChecked(true);
            if (currentKeyFileInfo.exists())
                pwdlg_ui->keyFileNameEdit->setText(currentKeyFileInfo.absoluteFilePath());
        }
        else
            pwdlg_ui->rememberKeyFileCheckBox->setChecked(false);
    }
    else
        //By default, the "Remember key file" check box is not checked
        settings.setValue(currentSetting, false);
    //  -- "Use password" setting --
    currentSetting = "Interface/UsePassword";
    if (settings.contains(currentSetting))
    {
        if (settings.value(currentSetting).toBool())
        {
            pwdlg_ui->enterThePasswordRadioButton->click();
            pwdlg_ui->firstPasswordEdit->setFocus();
        }
        else
            pwdlg_ui->useKeyFileRadioButton->click();
    }
    else
    {
        //By default, the "Use password" radio button is checked
        settings.setValue(currentSetting, true);
        pwdlg_ui->enterThePasswordRadioButton->click();
        pwdlg_ui->firstPasswordEdit->setFocus();
    }

}

PasswordDialog::~PasswordDialog()
{
    delete pwdlg_ui;
}

void PasswordDialog::accept()
{
    if (bUseSecondPwd)
    {
        if (pwdlg_ui->firstPasswordEdit->text()!= pwdlg_ui->secondPasswordEdit->text())
        {
            QMessageBox::warning(this, "Qccrypt",
                   "<qt>"+tr("The confirmation password was not the same as the first one. "
                   "Re-enter the confirmation or both.")+"</qt>",
                   QMessageBox::Ok | QMessageBox::Default, QMessageBox::NoButton);
            return;
        }
    }
    if (pwdlg_ui->enterThePasswordRadioButton->isChecked())
    {
        password = pwdlg_ui->firstPasswordEdit->text();
    }
    else
    {
        QFile file(currentKeyFileInfo.absoluteFilePath());
        if(!file.open(QIODevice::ReadOnly))
        {
            QMessageBox::critical(0, "Error", file.errorString());
            return;
        }
        else
        {
            QTextStream in(&file);
            //QString line;
            if (!in.atEnd())
            {
                //line = in.readLine();
                //password = line;
                file.close();
            }
            else
            {
                QMessageBox::warning(0, "Error", "This key file seems empty. Please select another one.");
                file.close();
                return;
            }
        }
    }

    QDialog::accept();
}

/*!
    Grabs the first password edit. Should be connected to the gotFocus() signal of the password
    lineedit widget.
*/
void PasswordDialog::grabFirstPassword()
{
    pwdlg_ui->firstPasswordEdit->grabKeyboard();
}


/*!
    Grabs the second password edit. Should be connected to the gotFocus() signal of the password
    lineedit widget.
*/
void PasswordDialog::grabSecondPassword()
{
    pwdlg_ui->secondPasswordEdit->grabKeyboard();
}


/*!
    Releases the password edit. Should be connected to the lostFocus() signal of the password
    lineedit widget
*/
void PasswordDialog::release()
{
    QWidget* widget = keyboardGrabber();
    if (widget)
        widget->releaseKeyboard();
}

/*!
    Returns the (new) password the user has entered.
*/
QString PasswordDialog::getPassword() const
{
    return password;
}

/*!
    Returns the (new) key file the user has entered.
*/
QFileInfo PasswordDialog::getKeyFile() const
{
    return currentKeyFileInfo;
}

/*!
    Returns true if the "Use temporary file" option is checked, false otherwise.
*/
bool PasswordDialog::isUseTemporaryFileOptionChecked()
{
    return pwdlg_ui->useTemporaryFileCheckBox->isChecked();
}

/*!
    Returns true if the "Use key file" option is checked, false otherwise.
*/
bool PasswordDialog::isUseKeyFileOptionChecked()
{
    return pwdlg_ui->useKeyFileRadioButton->isChecked();
}

/*!
    Checks the user's input and enables the Ok button if necessary. This slot is called always
    if the users changes the input.
*/
void PasswordDialog::checkOkEnabled() const
{
    if (pwdlg_ui->enterThePasswordRadioButton->isChecked())
    {
        if (bUseSecondPwd)
        {
            if (pwdlg_ui->firstPasswordEdit->hasAcceptableInput() && pwdlg_ui->secondPasswordEdit->hasAcceptableInput()
                    && pwdlg_ui->firstPasswordEdit->text().length() == pwdlg_ui->secondPasswordEdit->text().length())
                pwdlg_ui->okButton->setEnabled(true);
            else
                pwdlg_ui->okButton->setEnabled(false);
        }
        else
        {
            if (pwdlg_ui->firstPasswordEdit->hasAcceptableInput())
                pwdlg_ui->okButton->setEnabled(true);
            else
                pwdlg_ui->okButton->setEnabled(false);
        }
        pwdlg_ui->rememberKeyFileCheckBox->setEnabled(false);
    }
    else
    {
        if (currentKeyFileInfo.exists())
            pwdlg_ui->okButton->setEnabled(true);
        else
            pwdlg_ui->okButton->setEnabled(false);
        pwdlg_ui->rememberKeyFileCheckBox->setEnabled(true);
    }
}

void PasswordDialog::selectKeyFile()
{
    QString fileName = QFileDialog::getOpenFileName(this,
                                tr("Select the key file"),
                                QString(currentKeyFileInfo.absoluteFilePath()),
                                tr("All Files (*)"));
    currentKeyFileInfo.setFile(fileName);
    pwdlg_ui->keyFileNameEdit->setText(currentKeyFileInfo.absoluteFilePath());
    if (currentKeyFileInfo.exists())
        pwdlg_ui->okButton->setEnabled(true);
    else
        pwdlg_ui->okButton->setEnabled(false);
}

void PasswordDialog::enterThePasswordRadioButtonClicked()
{
    pwdlg_ui->firstPasswordEdit->setEnabled(true);
    if (bUseSecondPwd)
        pwdlg_ui->secondPasswordEdit->setEnabled(true);
    pwdlg_ui->showPasswordCheckBox->setEnabled(true);
    pwdlg_ui->rememberPasswordCheckBox->setEnabled(true);
    pwdlg_ui->browseKeyFileButton->setEnabled(false);
    settings.setValue("Interface/UsePassword", true);
    checkOkEnabled();
}

void PasswordDialog::useKeyFileRadioButtonClicked()
{
    pwdlg_ui->firstPasswordEdit->setEnabled(false);
    if (bUseSecondPwd)
        pwdlg_ui->secondPasswordEdit->setEnabled(false);
    pwdlg_ui->showPasswordCheckBox->setEnabled(false);
    pwdlg_ui->rememberPasswordCheckBox->setEnabled(false);
    pwdlg_ui->browseKeyFileButton->setEnabled(true);
    settings.setValue("Interface/UsePassword", false);
    checkOkEnabled();
}

void PasswordDialog::showPasswordCheckBoxClicked()
{
    QString currentSetting = "Interface/ShowPassword";
    if (pwdlg_ui->showPasswordCheckBox->isChecked())
    {
        bUseSecondPwd = false;
        pwdlg_ui->firstPasswordEdit->setEchoMode(QLineEdit::Normal);
        pwdlg_ui->secondPasswordEdit->setText("");
        pwdlg_ui->secondPasswordEdit->setEnabled(false);
        settings.setValue(currentSetting, true);
    }
    else
    {
        bUseSecondPwd = true;
        pwdlg_ui->firstPasswordEdit->setEchoMode(QLineEdit::Password);
        pwdlg_ui->secondPasswordEdit->setEnabled(true);
        pwdlg_ui->secondPasswordEdit->setEchoMode(QLineEdit::Password);
        pwdlg_ui->secondPasswordEdit->setText(pwdlg_ui->firstPasswordEdit->text());
        settings.setValue(currentSetting, false);
    }
}

void PasswordDialog::rememberPasswordCheckBoxClicked()
{
    QString currentSetting = "Interface/RememberPassword";
    if (pwdlg_ui->rememberPasswordCheckBox->isChecked())
        settings.setValue(currentSetting, true);
    else
        settings.setValue(currentSetting, false);
}

void PasswordDialog::useTemporaryFileCheckBoxClicked()
{
    QString currentSetting = "Interface/UseTemporaryFile";
    if (pwdlg_ui->useTemporaryFileCheckBox->isChecked())
        settings.setValue(currentSetting, true);
    else
        settings.setValue(currentSetting, false);
}

void PasswordDialog::rememberKeyFileCheckBoxClicked()
{
    QString currentSetting = "Interface/RememberKeyFile";
    if (pwdlg_ui->rememberKeyFileCheckBox->isChecked())
        settings.setValue(currentSetting, true);
    else
        settings.setValue(currentSetting, false);
}
