/*
* This file is part of Qccrypt, an open-source cross-platform frontend for ccrypt
* encryption tool.
* Copyright (C) 2006-2022  Philippe Beaureilles
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
* Contact e-mail: Philippe Beaureilles <qccrypt@free.fr>
* Program URL   : http://qccrypt.free.fr/
*
*/

#include "ccdlg.h"
#include "passworddialog.h"
#include "version.h"

#define round_up(n, k) (((n) - 1 + (k)) / (k)) //Thanks to Sebastian Lisken for this tip

CCryptDialog::CCryptDialog(const QString &fileName, QString sessionPassword, QWidget *parent)
    : QDialog(parent)
{
    ccdlg_ui.setupUi(this);

    setAcceptDrops(true);

	connect(ccdlg_ui.encryptOpenFileNameButton, SIGNAL(clicked()),
            this, SLOT(selectFileToEncrypt()));
	connect(ccdlg_ui.encryptFileButton, SIGNAL(clicked()),
			this, SLOT(encryptFile()));
	connect(ccdlg_ui.decryptOpenFileNameButton, SIGNAL(clicked()),
            this, SLOT(selectFileToDecrypt()));
	connect(ccdlg_ui.decryptFileButton, SIGNAL(clicked()),
			this, SLOT(decryptFile()));
	connect(ccdlg_ui.aboutButton, SIGNAL(clicked()),
			this, SLOT(about()));
	connect(ccdlg_ui.encryptFileRadioButton, SIGNAL(clicked()),
			this, SLOT(showEncryptFileGroupBox()));
	connect(ccdlg_ui.encryptFolderRadioButton, SIGNAL(clicked()),
			this, SLOT(showEncryptFolderGroupBox()));
	connect(ccdlg_ui.decryptFileRadioButton, SIGNAL(clicked()),
			this, SLOT(showDecryptFileGroupBox()));
	connect(ccdlg_ui.decryptFolderRadioButton, SIGNAL(clicked()),
			this, SLOT(showDecryptFolderGroupBox()));
	connect(ccdlg_ui.encryptFolderButton, SIGNAL(clicked()),
			this, SLOT(encryptFolder()));
	connect(ccdlg_ui.decryptFolderButton, SIGNAL(clicked()),
			this, SLOT(decryptFolder()));
	connect(ccdlg_ui.encryptOpenFolderNameButton, SIGNAL(clicked()),
            this, SLOT(selectFolderToEncrypt()));
	connect(ccdlg_ui.decryptOpenFolderNameButton, SIGNAL(clicked()),
            this, SLOT(selectFolderToDecrypt()));
    connect(ccdlg_ui.clearProgressListButton, SIGNAL(clicked()),
            this, SLOT(clearProgressList()));
    connect(ccdlg_ui.exitButton, SIGNAL(clicked()),
            this, SLOT(exitQccrypt()));
    connect(ccdlg_ui.stayOnTopCheckBox, SIGNAL(clicked()),
            this, SLOT(clickStayOnTopCheckbox()));
    connect(ccdlg_ui.keepExistingFileCheckBox, SIGNAL(clicked()),
            this, SLOT(clickKeepExistingFileDuringFileEncryptionCheckBox()));
    connect(ccdlg_ui.encryptFolderDoNotEncryptAlreadyEncryptedFilesCheckBox, SIGNAL(clicked()),
            this, SLOT(clickDoNotEncryptAlreadyEncryptedFilesDuringFolderEncryptionCheckBox()));
    connect(ccdlg_ui.keepExistingEncryptedFileCheckBox, SIGNAL(clicked()),
            this, SLOT(clickKeepExistingEncryptedFileDuringFileDecryptionCheckBox()));
    connect(ccdlg_ui.tabQccryptMode, SIGNAL(currentChanged(int)),
            this, SLOT(clickTabQccryptMode(int)));

    showEncryptFileGroupBox();
	showDecryptFileGroupBox();
	emptyAllFields();

    alreadyEncryptedFilesCount = 0;
    successfullyEncryptedFilesCount = 0;
    fileEncryptionErrorCount = 0;
    successfullyDecryptedFilesCount = 0;
    fileDecryptionErrorCount = 0;
	
	if (!fileName.isEmpty())
        determineAction(QString(fileName));

    //Session password
    sessionPwd = sessionPassword;

    //Read application's settings
    QString currentSetting;
    //  -- Stay On Top setting --
    currentSetting = "Interface/StayOnTop";
    Qt::WindowFlags flags = windowFlags();
    this->setFixedSize(this->size());
    if (settings.contains(currentSetting))
    {
        if (settings.value(currentSetting).toBool())
        {
            ccdlg_ui.stayOnTopCheckBox->setChecked(true);
            flags |= Qt::WindowMinimizeButtonHint | Qt::WindowStaysOnTopHint;
            setWindowFlags(flags);
        }
        else
        {
            flags  &= ~Qt::WindowStaysOnTopHint;
            setWindowFlags(flags);
            show();
        }
    }
    else
    {
        ccdlg_ui.stayOnTopCheckBox->setChecked(true);
        settings.setValue(currentSetting, true);
        flags |= Qt::WindowMinimizeButtonHint | Qt::WindowStaysOnTopHint;
        setWindowFlags(flags);
    }
    //  -- File Encryption / Keep existing file checkbox --
    currentSetting = "Encryption/KeepExistingFileDuringFileEncryption";
    if (settings.contains(currentSetting))
    {
        if (settings.value(currentSetting).toBool())
        {
            ccdlg_ui.keepExistingFileCheckBox->setChecked(true);
        }
        else
        {
            ccdlg_ui.keepExistingFileCheckBox->setChecked(false);
        }
    }
    else
    {
        //Setting not found. Unchecked by default.
        ccdlg_ui.keepExistingFileCheckBox->setChecked(false);
        //Save the setting
        settings.setValue(currentSetting, false);
    }
    //  -- Folder Encryption / Do Not Encrypt Already Encrypted Files checkbox --
    currentSetting = "Encryption/DoNotEncryptAlreadyEncryptedFilesDuringFolderEncryption";
    if (settings.contains(currentSetting))
    {
        if (settings.value(currentSetting).toBool())
        {
            ccdlg_ui.encryptFolderDoNotEncryptAlreadyEncryptedFilesCheckBox->setChecked(true);
        }
        else
        {
            ccdlg_ui.encryptFolderDoNotEncryptAlreadyEncryptedFilesCheckBox->setChecked(false);
        }
    }
    else
    {
        //Setting not found. Checked by default.
        ccdlg_ui.encryptFolderDoNotEncryptAlreadyEncryptedFilesCheckBox->setChecked(true);
        //Save the setting
        settings.setValue(currentSetting, true);
    }
    //  -- File Decryption / Keep Existing File During File Decryption checkbox --
    currentSetting = "Decryption/KeepExistingEncryptedFileDuringFileDecryption";
    if (settings.contains(currentSetting))
    {
        if (settings.value(currentSetting).toBool())
        {
            ccdlg_ui.keepExistingEncryptedFileCheckBox->setChecked(true);
        }
        else
        {
            ccdlg_ui.keepExistingEncryptedFileCheckBox->setChecked(false);
        }
    }
    else
    {
        //Setting not found. Unchecked by default.
        ccdlg_ui.keepExistingEncryptedFileCheckBox->setChecked(false);
        //Save the setting
        settings.setValue(currentSetting, false);
    }

    //Define application's title
    setWindowTitle("Qccrypt " + tr("%1").arg(QCCRYPT_CURRENT_VERSION));
    //Display the dialog
    show();
}

///////////////////////////////
// Display functions
///////////////////////////////
void CCryptDialog::updateFileToEncryptProperties()
{
    if (currentFileToEncryptInfo.exists())
	{
        ccdlg_ui.tabQccryptMode->setCurrentWidget(ccdlg_ui.encryptionTab);
        ccdlg_ui.encryptFileNameEdit->setText(currentFileToEncryptInfo.fileName());
        ccdlg_ui.encryptFilePathValueLabel->setText(currentFileToEncryptInfo.path());
        qlonglong size = round_up(currentFileToEncryptInfo.size(), 1024);
        ccdlg_ui.encryptFileSizeValueLabel->setText(tr("%1 K").arg(size));
        ccdlg_ui.encryptFileLastReadValueLabel->setText(currentFileToEncryptInfo.lastRead().toString());
        ccdlg_ui.encryptFileLastModValueLabel->setText(currentFileToEncryptInfo.lastModified().toString());
	    ccdlg_ui.encryptFileButton->setEnabled ( true );
	    ccdlg_ui.keepExistingFileCheckBox->setEnabled ( true );
        ccdlg_ui.encryptFilePathValueLabel->setToolTip(currentFileToEncryptInfo.path());
   	}
   	else
   	{
   		emptyEncryptFileFields();
        ccdlg_ui.encryptFilePathValueLabel->setToolTip(QString());
  	}
}

void CCryptDialog::updateFileToDecryptProperties()
{
    if (currentFileToDecryptInfo.exists())
	{
		ccdlg_ui.tabQccryptMode->setCurrentWidget(ccdlg_ui.decryptionTab);
        ccdlg_ui.decryptFileNameEdit->setText(currentFileToDecryptInfo.fileName());
        ccdlg_ui.decryptFilePathValueLabel->setText(currentFileToDecryptInfo.path());
        qlonglong size = round_up(currentFileToDecryptInfo.size(), 1024);
		ccdlg_ui.decryptFileSizeValueLabel->setText(tr("%1 K").arg(size));
        ccdlg_ui.decryptFileLastReadValueLabel->setText(currentFileToDecryptInfo.lastRead().toString());
        ccdlg_ui.decryptFileLastModValueLabel->setText(currentFileToDecryptInfo.lastModified().toString());
	    ccdlg_ui.decryptFileButton->setEnabled ( true );
	    ccdlg_ui.keepExistingEncryptedFileCheckBox->setEnabled ( true );
        ccdlg_ui.decryptFilePathValueLabel->setToolTip(currentFileToDecryptInfo.path());
   	}
   	else
   	{
   		emptyDecryptFileFields();
        ccdlg_ui.decryptFilePathValueLabel->setToolTip(QString());
  	}
}

void CCryptDialog::updateFolderToEncryptProperties()
{
    if (currentFolderToEncryptInfo.exists())
	{
		ccdlg_ui.tabQccryptMode->setCurrentWidget(ccdlg_ui.encryptionTab);
        ccdlg_ui.encryptFolderNameEdit->setText(currentFolderToEncryptInfo.path());
        ccdlg_ui.encryptFolderSizeValueLabel->setText(tr("%1 Kbytes").arg(round_up(getFolderSize(currentFolderToEncryptInfo.path()),1024)));
        ccdlg_ui.encryptFolderSubFoldersCountValueLabel->setText(tr("%1").arg(getSubFoldersCount(currentFolderToEncryptInfo.path())));
        subFilesCount = getSubFilesCount(currentFolderToEncryptInfo.path());
        ccdlg_ui.encryptFolderSubFilesCountValueLabel->setText(tr("%1").arg(subFilesCount));
        ccdlg_ui.encryptFolderButton->setEnabled ( true );
        ccdlg_ui.encryptFolderNameEdit->setToolTip(currentFolderToEncryptInfo.path());
	}
	else
	{
   		emptyEncryptFolderFields();
        ccdlg_ui.encryptFolderNameEdit->setToolTip(QString());
	}
}

void CCryptDialog::updateFolderToDecryptProperties()
{
    if (currentFolderToDecryptInfo.exists())
	{
		ccdlg_ui.tabQccryptMode->setCurrentWidget(ccdlg_ui.decryptionTab);
        ccdlg_ui.decryptFolderNameEdit->setText(currentFolderToDecryptInfo.path());
        ccdlg_ui.decryptFolderSizeValueLabel->setText(tr("%1 Kbytes").arg(round_up(getFolderSize(currentFolderToDecryptInfo.path()),1024)));
        ccdlg_ui.decryptFolderSubFoldersCountValueLabel->setText(tr("%1").arg(getSubFoldersCount(currentFolderToDecryptInfo.path())));
        subFilesCount = getSubFilesCount(currentFolderToDecryptInfo.path());
        ccdlg_ui.decryptFolderSubFilesCountValueLabel->setText(tr("%1").arg(subFilesCount));
        ccdlg_ui.decryptFolderButton->setEnabled ( true );
        ccdlg_ui.decryptFolderNameEdit->setToolTip(currentFolderToDecryptInfo.path());
	}
	else
	{
   		emptyDecryptFolderFields();
        ccdlg_ui.decryptFolderNameEdit->setToolTip(QString());
	}
}

void CCryptDialog::emptyEncryptFileFields()
{
	ccdlg_ui.encryptFileNameEdit->setText("");
	ccdlg_ui.encryptFilePathValueLabel->setText("");
	ccdlg_ui.encryptFileSizeValueLabel->setText("");
	ccdlg_ui.encryptFileLastReadValueLabel->setText("");
    ccdlg_ui.encryptFileLastModValueLabel->setText("");
	ccdlg_ui.encryptFileButton->setEnabled ( false );
    ccdlg_ui.keepExistingFileCheckBox->setEnabled ( false );
}

void CCryptDialog::emptyDecryptFileFields()
{
	ccdlg_ui.decryptFileNameEdit->setText("");
	ccdlg_ui.decryptFilePathValueLabel->setText("");
	ccdlg_ui.decryptFileSizeValueLabel->setText("");
	ccdlg_ui.decryptFileLastReadValueLabel->setText("");
    ccdlg_ui.decryptFileLastModValueLabel->setText("");
	ccdlg_ui.decryptFileButton->setEnabled ( false );
    ccdlg_ui.keepExistingEncryptedFileCheckBox->setEnabled ( false );
}

void CCryptDialog::emptyEncryptFolderFields()
{
	ccdlg_ui.encryptFolderNameEdit->setText("");
	ccdlg_ui.encryptFolderSizeValueLabel->setText("");
	ccdlg_ui.encryptFolderSubFoldersCountValueLabel->setText("");
    ccdlg_ui.encryptFolderSubFilesCountValueLabel->setText("");
	ccdlg_ui.encryptFolderButton->setEnabled ( false );
}

void CCryptDialog::emptyDecryptFolderFields()
{
	ccdlg_ui.decryptFolderNameEdit->setText("");
	ccdlg_ui.decryptFolderSizeValueLabel->setText("");
	ccdlg_ui.decryptFolderSubFoldersCountValueLabel->setText("");
    ccdlg_ui.decryptFolderSubFilesCountValueLabel->setText("");
	ccdlg_ui.decryptFolderButton->setEnabled ( false );
}

void CCryptDialog::emptyAllFields()
{
	emptyEncryptFileFields();
	emptyEncryptFolderFields();
	emptyDecryptFileFields();
	emptyDecryptFolderFields();
}

void CCryptDialog::showEncryptFileGroupBox()
{
	ccdlg_ui.encryptFolderGroupBox->hide();
	ccdlg_ui.encryptFileGroupBox->show();
    ccdlg_ui.encryptFileInformationGroupBox->show();
    ccdlg_ui.encryptFolderInformationGroupBox->hide();
    //Hide all action buttons, except the "Encrypt File" button
    ccdlg_ui.decryptFolderButton->setEnabled(false);
    ccdlg_ui.encryptFolderButton->setEnabled(false);
    if (ccdlg_ui.encryptFileNameEdit->text() != "")
        ccdlg_ui.encryptFileButton->setEnabled(true);
    ccdlg_ui.decryptFileButton->setEnabled(false);
}

void CCryptDialog::showDecryptFileGroupBox()
{
	ccdlg_ui.decryptFolderGroupBox->hide();
	ccdlg_ui.decryptFileGroupBox->show();
    ccdlg_ui.decryptFileInformationGroupBox->show();
    ccdlg_ui.decryptFolderInformationGroupBox->hide();
    //Hide all action buttons, except the "Decrypt File" button
    ccdlg_ui.decryptFolderButton->setEnabled(false);
    ccdlg_ui.encryptFolderButton->setEnabled(false);
    ccdlg_ui.encryptFileButton->setEnabled(false);
    if (ccdlg_ui.decryptFileNameEdit->text() != "")
        ccdlg_ui.decryptFileButton->setEnabled(true);
}

void CCryptDialog::showEncryptFolderGroupBox()
{
	ccdlg_ui.encryptFileGroupBox->hide();
	ccdlg_ui.encryptFolderGroupBox->show();
    ccdlg_ui.encryptFileInformationGroupBox->hide();
    ccdlg_ui.encryptFolderInformationGroupBox->show();
    //Hide all action buttons, except the "Encrypt Folder" button
    ccdlg_ui.decryptFolderButton->setEnabled(false);
    if (ccdlg_ui.encryptFolderNameEdit->text() != "")
        ccdlg_ui.encryptFolderButton->setEnabled(true);
    ccdlg_ui.encryptFileButton->setEnabled(false);
    ccdlg_ui.decryptFileButton->setEnabled(false);
}

void CCryptDialog::showDecryptFolderGroupBox()
{
	ccdlg_ui.decryptFileGroupBox->hide();
	ccdlg_ui.decryptFolderGroupBox->show();
    ccdlg_ui.decryptFileInformationGroupBox->hide();
    ccdlg_ui.decryptFolderInformationGroupBox->show();
    //Hide all action buttons, except the "Decrypt Folder" button
    if (ccdlg_ui.decryptFolderNameEdit->text() != "")
        ccdlg_ui.decryptFolderButton->setEnabled(true);
    ccdlg_ui.encryptFolderButton->setEnabled(false);
    ccdlg_ui.encryptFileButton->setEnabled(false);
    ccdlg_ui.decryptFileButton->setEnabled(false);
}

qlonglong CCryptDialog::getFolderSize(QString path)
{
	qlonglong size=0;
	QDir folder;
	folder.setPath(path);
	QFileInfoList fileInfoList = folder.entryInfoList(QDir::AllEntries|QDir::NoDotAndDotDot);
	for(int i = 0; i < fileInfoList.size(); ++i)
	{
		if(fileInfoList.at(i).isDir())
			size+=getFolderSize(fileInfoList.at(i).path() + "/" + fileInfoList.at(i).fileName());
		else
	  		size+=fileInfoList.at(i).size();
	}
	return size;
}

qlonglong CCryptDialog::getSubFoldersCount(QString path)
{
	qlonglong count=0;
	QDir folder;
	folder.setPath(path);
	QFileInfoList fileInfoList = folder.entryInfoList(QDir::AllEntries|QDir::NoDotAndDotDot);
	for(int i = 0; i < fileInfoList.size(); ++i)
	{
		if(fileInfoList.at(i).isDir())
		{
			count++;
            count+=getSubFoldersCount(fileInfoList.at(i).path() + "/" + fileInfoList.at(i).fileName());
		}
	}
	return count;
}

qlonglong CCryptDialog::getSubFilesCount(QString path)
{
	qlonglong count=0;
	QDir folder;
	folder.setPath(path);
	QFileInfoList fileInfoList = folder.entryInfoList(QDir::AllEntries|QDir::NoDotAndDotDot);
	for(int i = 0; i < fileInfoList.size(); ++i)
	{
		if (fileInfoList.at(i).isFile())
			count++;
		if (fileInfoList.at(i).isDir())
            count+=getSubFilesCount(fileInfoList.at(i).path() + "/" + fileInfoList.at(i).fileName());
	}
	return count;
}

///////////////////////////////
// Connectors
///////////////////////////////
void CCryptDialog::selectFileToEncrypt()
{
	QString fileName = QFileDialog::getOpenFileName(this,
								tr("Select the file to encrypt"),
                                QString(currentFileToEncryptInfo.absoluteFilePath()),
								tr("All Files (*)"));

    currentFileToEncryptInfo.setFile(fileName);
	updateFileToEncryptProperties();
}

void CCryptDialog::encryptFile()
{
    PasswordDialog pwDlg(true, sessionPwd, sessionKeyFile, this);
	if (pwDlg.exec() == QDialog::Accepted)
	{
		QString password = pwDlg.getPassword();
        sessionKeyFile = pwDlg.getKeyFile();
        bool bUseTempFile = pwDlg.isUseTemporaryFileOptionChecked();
        bool bUseKeyFile = pwDlg.isUseKeyFileOptionChecked();

        if (bUseKeyFile)
        {
            // Use a key file
            if (sessionKeyFile.exists())
            {
                QString targetFileName;
                targetFileName = QString(currentFileToEncryptInfo.absoluteFilePath());
                QFileInfo targetFile;
                targetFile = QFileInfo(targetFileName  + ".cpt");

                if (targetFile.isFile())
                {
                    int reply = QMessageBox::warning(this, tr("Replace file confirmation"),
                                    tr("The target file already exists. Do you want to overwrite it ?"),
                                    tr("&Yes"),
                                    tr("&No"));
                    if (reply == 1)
                        return;
                }

                ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Start of task: encrypt file \"" + targetFileName + "\" ***********");
                ccdlg_ui.progressBar->reset();
                ccdlg_ui.progressBar->setMinimum(0);
                ccdlg_ui.progressBar->setMaximum(1);
                currentProgress = 0;
                ccdlg_ui.progressBar->setValue(currentProgress);
                int resultCode = runFileEncryption(targetFileName, "", bool((ccdlg_ui.encryptFolderDoNotEncryptAlreadyEncryptedFilesCheckBox->isChecked())), bUseTempFile, bUseKeyFile);
                ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - End of task: encrypt file \"" + targetFileName + "\" ***********");
                ccdlg_ui.progressList->scrollToBottom();
                currentProgress++;
                ccdlg_ui.progressBar->setValue(currentProgress);

                switch (resultCode) {
                    case 0 :
                        QMessageBox::information(this, "Qccrypt", "Encryption successful.", "&Ok");
                        currentFileToEncryptInfo.setFile("");
                        updateFileToEncryptProperties();
                        break;
                    default :
                        QMessageBox::critical(this, "Qccrypt", "Encryption error.", "&Ok");
                }
            }
            else
            {
                // The key file does not exist anymore
                QMessageBox::critical(this, "Qccrypt", "The key file does not exist anymore! Unable to continue.", "&Ok");
            }

        }
        else
        {
            // Do not use a key file
            if (!password.isEmpty())
            {
                sessionPwd = password;
                QString targetFileName;
                targetFileName = QString(currentFileToEncryptInfo.absoluteFilePath());
                QFileInfo targetFile;
                targetFile = QFileInfo(targetFileName  + ".cpt");

                if (targetFile.isFile())
                {
                    int reply = QMessageBox::warning(this, tr("Replace file confirmation"),
                                    tr("The target file already exists. Do you want to overwrite it ?"),
                                    tr("&Yes"),
                                    tr("&No"));
                    if (reply == 1)
                        return;
                }

                ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Start of task: encrypt file \"" + targetFileName + "\" ***********");
                ccdlg_ui.progressBar->reset();
                ccdlg_ui.progressBar->setMinimum(0);
                ccdlg_ui.progressBar->setMaximum(1);
                currentProgress = 0;
                ccdlg_ui.progressBar->setValue(currentProgress);
                int resultCode = runFileEncryption(targetFileName, password, bool((ccdlg_ui.encryptFolderDoNotEncryptAlreadyEncryptedFilesCheckBox->isChecked())), bUseTempFile, bUseKeyFile);
                ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - End of task: encrypt file \"" + targetFileName + "\" ***********");
                ccdlg_ui.progressList->scrollToBottom();
                currentProgress++;
                ccdlg_ui.progressBar->setValue(currentProgress);

                switch (resultCode) {
                    case 0 :
                        QMessageBox::information(this, "Qccrypt", "Encryption successful.", "&Ok");
                        currentFileToEncryptInfo.setFile("");
                        updateFileToEncryptProperties();
                        break;
                    default :
                        QMessageBox::critical(this, "Qccrypt", "Encryption error.", "&Ok");
                }
            }
            else
                QMessageBox::critical(this, "Qccrypt", "You must enter a password to encrypt this file!", "&Ok");
            // End
        }
	}
}

void CCryptDialog::selectFileToDecrypt()
{
	QString fileName = QFileDialog::getOpenFileName(this,
								tr("Select the file to decrypt"),
                                QString(currentFileToDecryptInfo.absoluteFilePath()),
								tr("ccrypt crypted file (*.cpt)"));
    currentFileToDecryptInfo.setFile(fileName);
	updateFileToDecryptProperties();
}

void CCryptDialog::decryptFile()
{
    PasswordDialog pwDlg(false, sessionPwd, sessionKeyFile, this);
    if (pwDlg.exec() == QDialog::Accepted)
    {
        QString password = pwDlg.getPassword();
        sessionKeyFile = pwDlg.getKeyFile();
        bool bUseTempFile = pwDlg.isUseTemporaryFileOptionChecked();
        bool bUseKeyFile = pwDlg.isUseKeyFileOptionChecked();

        if (bUseKeyFile)
        {
            // Use a key file
            if (sessionKeyFile.exists())
            {
                QString targetFileName;
                targetFileName = QString(currentFileToDecryptInfo.absoluteFilePath());
                targetFileName = targetFileName.left(targetFileName.length() - 4);
                QFileInfo targetFile;
                targetFile = QFileInfo(targetFileName);

                if (targetFile.isFile())
                {
                    int reply = QMessageBox::warning(this, tr("Replace file confirmation"),
                                    tr("The target file already exists. Do you want to overwrite it ?"),
                                    tr("&Yes"),
                                    tr("&No"));
                    if (reply == 1)
                        return;
                }

                ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Start of task: decrypt file \"" + targetFileName + "\" ***********");
                ccdlg_ui.progressBar->reset();
                ccdlg_ui.progressBar->setMinimum(0);
                ccdlg_ui.progressBar->setMaximum(1);
                currentProgress = 0;
                ccdlg_ui.progressBar->setValue(currentProgress);
                int resultCode = runFileDecryption(currentFileToDecryptInfo.absoluteFilePath(), password, bUseTempFile, bUseKeyFile);
                ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - End of task: decrypt file \"" + targetFileName + "\" ***********");
                ccdlg_ui.progressList->scrollToBottom();
                currentProgress++;
                ccdlg_ui.progressBar->setValue(currentProgress);

                switch (resultCode) {
                    case 0 :
                        QMessageBox::information(this, "Qccrypt", "Decryption successful.", "&Ok");
                        currentFileToDecryptInfo.setFile("");
                        updateFileToDecryptProperties();
                        break;
                    case 4 :
                        QMessageBox::critical(this, "Qccrypt", "The password specified was incorrect!", "&Ok");
                        decryptFile();
                        break;
                    default :
                        QMessageBox::critical(this, "Qccrypt", "Decryption error.", "&Ok");
                }
            }
            else
                QMessageBox::critical(this, "Qccrypt", "The key file was not found!\n\n" + sessionKeyFile.filePath(), "&Ok");
        }
        else
        {
            // Do not use a key file
            if (!password.isEmpty())
            {
                sessionPwd = password;
                QString targetFileName;
                targetFileName = QString(currentFileToDecryptInfo.absoluteFilePath());
                targetFileName = targetFileName.left(targetFileName.length() - 4);
                QFileInfo targetFile;
                targetFile = QFileInfo(targetFileName);

                if (targetFile.isFile())
                {
                    int reply = QMessageBox::warning(this, tr("Replace file confirmation"),
                                    tr("The target file already exists. Do you want to overwrite it ?"),
                                    tr("&Yes"),
                                    tr("&No"));
                    if (reply == 1)
                        return;
                }

                ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Start of task: decrypt file \"" + targetFileName + "\" ***********");
                ccdlg_ui.progressBar->reset();
                ccdlg_ui.progressBar->setMinimum(0);
                ccdlg_ui.progressBar->setMaximum(1);
                currentProgress = 0;
                ccdlg_ui.progressBar->setValue(currentProgress);
                int resultCode = runFileDecryption(currentFileToDecryptInfo.absoluteFilePath(), password, bUseTempFile, bUseKeyFile);
                ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - End of task: decrypt file \"" + targetFileName + "\" ***********");
                ccdlg_ui.progressList->scrollToBottom();
                currentProgress++;
                ccdlg_ui.progressBar->setValue(currentProgress);

                switch (resultCode) {
                    case 0 :
                        QMessageBox::information(this, "Qccrypt", "Decryption successful.", "&Ok");
                        currentFileToDecryptInfo.setFile("");
                        updateFileToDecryptProperties();
                        break;
                    case 4 :
                        QMessageBox::critical(this, "Qccrypt", "The password specified was incorrect!", "&Ok");
                        decryptFile();
                        break;
                    default :
                        QMessageBox::critical(this, "Qccrypt", "Decryption error.", "&Ok");
                }
            }
            else
                QMessageBox::critical(this, "Qccrypt", "You must enter the password to decrypt this file!", "&Ok");
        }
	}
}

void CCryptDialog::about()
{
    //Read ccrypt command line tool version from Stdout
    QString program = "ccrypt";
    QProcess process;
    QStringList arguments;

    arguments << "-V";
    process.start(program, arguments);
    process.waitForFinished(-1);
    QByteArray ccryptProcessOutput = process.readAllStandardOutput();
    QString ccryptVersionInfo = QString(ccryptProcessOutput);

    //Display the About message box
    QMessageBox aboutMsgBox(this);
    const char *qt_version = qVersion();
    aboutMsgBox.setWindowTitle("About Qccrypt " + tr("%1").arg(QCCRYPT_CURRENT_VERSION));
    aboutMsgBox.setTextFormat(Qt::RichText);   //this is what makes the links clickable
    aboutMsgBox.setText("Qccrypt is an Open Source frontend for ccrypt encryption/decryption tool.<br>It is written in C++ with Qt 5 libraries by Philippe Beaureilles.<br><br>Compiled with Qt Version " + tr(QT_VERSION_STR) +"<br>Qt Runtime Version: "+ tr(qt_version) +"<br><br>Ccrypt version:<br>"+ ccryptVersionInfo+"<br><br>This program is free software. You can redistribute it and/or modify it under the terms of\nthe GNU General Public License as published by the Free Software Foundation\neither version 2 of the License, or (at your option) any later version.<br><br>Special thanks for their contribution to Qccrypt to:<ul><li>Sebastian Lisken (errors report, corrections of language mistakes, portions of code and suggestions).</li><li>Asen Anastassov (suggestions and tests).</li><li>Shohei Kusakata (creation of the AUR package).</li></ul><br><center><a href='http://qccrypt.free.fr/'>http://qccrypt.free.fr/</a></center>");
    aboutMsgBox.setIconPixmap(QPixmap(":/ressources/images/qccrypt64.png"));
    aboutMsgBox.setStandardButtons(QMessageBox::Ok);
    aboutMsgBox.exec();
}

int CCryptDialog::runFileEncryption(QString fileName, QString password, bool bEncryptAlreadyEncryptedFiles, bool bUseTemporaryFile, bool bUseKeyFile)
{
    QString program = "ccrypt";
    QStringList arguments;
    QProcess *encryptProcess = new QProcess(this);
    QFileInfo fileInfo;
    fileInfo = QFileInfo(fileName);
    QListWidgetItem *newItem = new QListWidgetItem;
    QTemporaryFile* tmpFile = new QTemporaryFile();

    if (bUseKeyFile)
    {
        // Use a key file
        delete tmpFile;
        if (bEncryptAlreadyEncryptedFiles && (fileInfo.suffix() == "cpt"))
        {
            ccdlg_ui.progressList->addItem(newItem);
            newItem->setText(QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Ignoring \"" + fileName + "\" (already encrypted)");
            ccdlg_ui.progressList->scrollToBottom();
            alreadyEncryptedFilesCount++;
            delete encryptProcess;
            return 0;
        }
        else
        {
            newItem->setText(QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Encryting \"" + fileName + "\"...");
            ccdlg_ui.progressList->addItem(newItem);
            ccdlg_ui.progressList->scrollToBottom();
            arguments << "-e" << "-f" << fileName << "-k" << sessionKeyFile.filePath();
            QApplication::setOverrideCursor(Qt::WaitCursor);
            encryptProcess->start(program, arguments);
            QApplication::processEvents();
            encryptProcess->waitForFinished(-1);
            QApplication::restoreOverrideCursor();

            switch (encryptProcess->exitCode ()) {
                    case 0 :
                            newItem->setText(newItem->text() + " OK!");
                            successfullyEncryptedFilesCount++;
                            break;
                    default :
                            newItem->setText(newItem->text() + " Failed!");
                            fileEncryptionErrorCount++;
            }
            return encryptProcess->exitCode ();
        }
    }
    else
    {
        //Do not use a key file
        if (bEncryptAlreadyEncryptedFiles && (fileInfo.suffix() == "cpt"))
        {
            ccdlg_ui.progressList->addItem(newItem);
            newItem->setText(QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Ignoring \"" + fileName + "\" (already encrypted)");
            ccdlg_ui.progressList->scrollToBottom();
            alreadyEncryptedFilesCount++;
            delete tmpFile;
            delete encryptProcess;
            return 0;
        }
        else
        {
            newItem->setText(QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Encryting \"" + fileName + "\"...");
            ccdlg_ui.progressList->addItem(newItem);
            ccdlg_ui.progressList->scrollToBottom();

            if (bUseTemporaryFile)
            {
                if ( tmpFile->open() )
                {
                    QByteArray baPassword = password.toLocal8Bit();
                    const char *csPassword = baPassword.data();
                    tmpFile->write( csPassword );
                    tmpFile->flush();
                }
                else
                {
                    QMessageBox::critical(this, "Qccrypt", "Error: unable to create the temporary file: " + tmpFile->fileName(), "&Ok");
                    delete tmpFile;
                    delete encryptProcess;
                    return 80;
                }
                if (ccdlg_ui.keepExistingFileCheckBox->isChecked())
                {
                    arguments << "-e" << "-f" << "-k" << tmpFile->fileName();
                    encryptProcess->setStandardInputFile(fileName);
                    encryptProcess->setStandardOutputFile(fileName + ".cpt");
                }
                else
                {
                    arguments << "-e" << "-f" << fileName << "-k" << tmpFile->fileName();
                }

            }
            else
            {
                // Temporay file not used
                if (ccdlg_ui.keepExistingFileCheckBox->isChecked())
                {
                    arguments << "-e" << "-f" << "-K" << password;
                    encryptProcess->setStandardInputFile(fileName);
                    encryptProcess->setStandardOutputFile(fileName + ".cpt");
                }
                else
                {
                    arguments << "-e" << "-f" << "-K" << password << fileName ;
                }
            }

            QApplication::setOverrideCursor(Qt::WaitCursor);
            encryptProcess->start(program, arguments);
            QApplication::processEvents();
            encryptProcess->waitForFinished(-1);
            if (bUseTemporaryFile)
            {
                // Write random data into the temporary file to prevent password recovery from deleted files
                int randomValue = QRandomGenerator::global()->bounded(100);
                QString randomString = getRandomString(password.length() + randomValue);
                QByteArray baRandomString = randomString.toLocal8Bit();
                const char *csRandomString = baRandomString.data();
                tmpFile->seek(0);
                tmpFile->write( csRandomString );
                tmpFile->flush();
            }
            delete tmpFile;
            QApplication::restoreOverrideCursor();

            switch (encryptProcess->exitCode ()) {
                    case 0 :
                            newItem->setText(newItem->text() + " OK!");
                            successfullyEncryptedFilesCount++;
                            break;
                    default :
                            newItem->setText(newItem->text() + " Failed!");
                            fileEncryptionErrorCount++;
            }

            return encryptProcess->exitCode ();
        }
    }
 }

int CCryptDialog::runFolderEncryption(QString folderName, QString password, bool bUseTemporaryFile, bool bUseKeyFile)
{
    QDir folderDir;
    folderDir.setPath(folderName);
    QFileInfoList fileInfoList = folderDir.entryInfoList(QDir::AllEntries|QDir::NoDotAndDotDot);

    for(int i = 0; i < fileInfoList.size(); ++i)
    {
        if (fileInfoList.at(i).isFile())
        {
            runFileEncryption(fileInfoList.at(i).absoluteFilePath(), password, bool((ccdlg_ui.encryptFolderDoNotEncryptAlreadyEncryptedFilesCheckBox->isChecked())), bUseTemporaryFile, bUseKeyFile);
            currentProgress++;
        }
        if (fileInfoList.at(i).isDir())
        {
            runFolderEncryption(fileInfoList.at(i).absoluteFilePath(), password, bUseTemporaryFile, bUseKeyFile);
        }
        ccdlg_ui.progressBar->setValue(currentProgress);
    }
    return 0;
}

int CCryptDialog::runFolderDecryption(QString folderName, QString password, bool bUseTemporaryFile, bool bUseKeyFile)
{
    QDir folderDir;
    folderDir.setPath(folderName);
    QFileInfoList fileInfoList = folderDir.entryInfoList(QDir::AllEntries|QDir::NoDotAndDotDot);
    for(int i = 0; i < fileInfoList.size(); ++i)
    {
        if (fileInfoList.at(i).isFile())
        {
            runFileDecryption(fileInfoList.at(i).absoluteFilePath(), password, bUseTemporaryFile, bUseKeyFile);
            currentProgress++;
        }
        if (fileInfoList.at(i).isDir())
        {
            runFolderDecryption(fileInfoList.at(i).absoluteFilePath(), password, bUseTemporaryFile, bUseKeyFile);
        }
        ccdlg_ui.progressBar->setValue(currentProgress);
    }
    return 0;
}


int CCryptDialog::runFileDecryption(QString fileName, QString password, bool bUseTemporaryFile, bool bUseKeyFile)
{
    QString program = "ccrypt";
    QStringList arguments;
    QProcess *encryptProcess = new QProcess(this);

    QListWidgetItem *newItem = new QListWidgetItem;
    QTemporaryFile* tmpFile = new QTemporaryFile();

    newItem->setText(QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Decryting \"" + fileName + "\"...");
    ccdlg_ui.progressList->addItem(newItem);
    ccdlg_ui.progressList->scrollToBottom();

    if (bUseKeyFile)
    {
        // Use a key file
        delete tmpFile;
        arguments << "-d" << "-f" << fileName << "-k" << sessionKeyFile.filePath();
        QApplication::setOverrideCursor(Qt::WaitCursor);
        encryptProcess->start(program, arguments);
        QApplication::processEvents();
        encryptProcess->waitForFinished();
        QApplication::restoreOverrideCursor();
        switch (encryptProcess->exitCode ()) {
            case 0 :
                newItem->setText(newItem->text() + " OK!");
                successfullyDecryptedFilesCount++;
                break;
            default :
                newItem->setText(newItem->text() + " Failed!");
                fileDecryptionErrorCount++;
        }

        return encryptProcess->exitCode ();
    }
    else
    {
        //Do not use a key file
        if (bUseTemporaryFile)
        {
            if ( tmpFile->open() )
            {
                QByteArray baPassword = password.toLocal8Bit();
                const char *csPassword = baPassword.data();
                tmpFile->write( csPassword );
                tmpFile->flush();
            }
            else
            {
                QMessageBox::critical(this, "Qccrypt", "Error: unable to create the temporary file: " + tmpFile->fileName(), "&Ok");
                delete tmpFile;
                delete encryptProcess;
                return 80;
            }
            if (ccdlg_ui.keepExistingEncryptedFileCheckBox->isChecked())
            {
                arguments << "-d" << "-f" << "-k" << tmpFile->fileName();
                encryptProcess->setStandardInputFile(fileName);
                QString newFileName = fileName;
                newFileName.truncate(fileName.length() - 4);
                encryptProcess->setStandardOutputFile(newFileName);
            }
            else
            {
                arguments << "-d" << "-f" << fileName << "-k" << tmpFile->fileName();
            }
        }
        else
        {// Do not use a temporary file
            if (ccdlg_ui.keepExistingEncryptedFileCheckBox->isChecked())
            {
                arguments << "-d" << "-f" << "-K" << password;
                encryptProcess->setStandardInputFile(fileName);
                QString newFileName = fileName;
                newFileName.truncate(fileName.length() - 4);
                encryptProcess->setStandardOutputFile(newFileName);
            }
            else
                arguments << "-d" << "-f" << "-K" << password << fileName ;
        }

        QApplication::setOverrideCursor(Qt::WaitCursor);
        encryptProcess->start(program, arguments);
        QApplication::processEvents();
        encryptProcess->waitForFinished();
        if (bUseTemporaryFile)
        {
            // Write random data into the temporary file to prevent password recovery from deleted files
            int randomValue = QRandomGenerator::global()->bounded(100);
            QString randomString = getRandomString(password.length() + randomValue);
            QByteArray baRandomString = randomString.toLocal8Bit();
            const char *csRandomString = baRandomString.data();
            tmpFile->seek(0);
            tmpFile->write( csRandomString );
            tmpFile->flush();
        }
        delete tmpFile;
        QApplication::restoreOverrideCursor();

        switch (encryptProcess->exitCode ()) {
            case 0 :
                newItem->setText(newItem->text() + " OK!");
                successfullyDecryptedFilesCount++;
                break;
            default :
                newItem->setText(newItem->text() + " Failed!");
                fileDecryptionErrorCount++;
        }

        return encryptProcess->exitCode ();
    }
}

void CCryptDialog::encryptFolder()
{
    PasswordDialog pwDlg(true, sessionPwd, sessionKeyFile, this);
	if (pwDlg.exec() == QDialog::Accepted)
	{
		QString password = pwDlg.getPassword();
        sessionKeyFile = pwDlg.getKeyFile();
        bool bUseTempFile = pwDlg.isUseTemporaryFileOptionChecked();
        bool bUseKeyFile = pwDlg.isUseKeyFileOptionChecked();

        if (bUseKeyFile)
        {
            // Use a key file
            if (sessionKeyFile.exists())
            {
                alreadyEncryptedFilesCount = 0;
                successfullyEncryptedFilesCount = 0;
                fileEncryptionErrorCount = 0;
                ccdlg_ui.progressBar->reset();
                ccdlg_ui.progressBar->setMinimum(0);
                ccdlg_ui.progressBar->setMaximum(subFilesCount);
                currentProgress = 0;
                ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Start of task: encrypt folder \"" + currentFolderToEncryptInfo.path() + "\" ***********");
                runFolderEncryption(currentFolderToEncryptInfo.path(), password, bUseTempFile, bUseKeyFile);
                ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - End of task: encrypt folder \"" + currentFolderToEncryptInfo.path() + "\" ***********");
                ccdlg_ui.progressList->scrollToBottom();
                QString sReportContent = "Encryption finished.\n";
                if (ccdlg_ui.encryptFolderDoNotEncryptAlreadyEncryptedFilesCheckBox->isChecked())
                    sReportContent += "\nAlready encrypted files: " + tr("%1").arg(alreadyEncryptedFilesCount);
                sReportContent += "\nSuccessfully encrypted files: " + tr("%1").arg(successfullyEncryptedFilesCount) + "\nFiles failed to encrypt: " + tr("%1").arg(fileEncryptionErrorCount);
                sReportContent += "\n\nTotal of files processed: " + tr("%1").arg(successfullyEncryptedFilesCount + fileEncryptionErrorCount);
                QMessageBox::information(this, "Qccrypt", sReportContent, "&Ok");
            }
            else
                QMessageBox::critical(this, "Qccrypt", "The key file was not found!\n\n" + sessionKeyFile.filePath(), "&Ok");
        }
        else
        {
            // Do not use a key file
            if (!password.isEmpty())
            {
                sessionPwd = password;
                alreadyEncryptedFilesCount = 0;
                successfullyEncryptedFilesCount = 0;
                fileEncryptionErrorCount = 0;
                ccdlg_ui.progressBar->reset();
                ccdlg_ui.progressBar->setMinimum(0);
                ccdlg_ui.progressBar->setMaximum(subFilesCount);
                currentProgress = 0;
                ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Start of task: encrypt folder \"" + currentFolderToEncryptInfo.path() + "\" ***********");
                runFolderEncryption(currentFolderToEncryptInfo.path(), password, bUseTempFile, bUseKeyFile);
                ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - End of task: encrypt folder \"" + currentFolderToEncryptInfo.path() + "\" ***********");
                ccdlg_ui.progressList->scrollToBottom();
                QString sReportContent = "Encryption finished.\n";
                if (ccdlg_ui.encryptFolderDoNotEncryptAlreadyEncryptedFilesCheckBox->isChecked())
                    sReportContent += "\nAlready encrypted files: " + tr("%1").arg(alreadyEncryptedFilesCount);
                sReportContent += "\nSuccessfully encrypted files: " + tr("%1").arg(successfullyEncryptedFilesCount) + "\nFiles failed to encrypt: " + tr("%1").arg(fileEncryptionErrorCount);
                sReportContent += "\n\nTotal of files processed: " + tr("%1").arg(successfullyEncryptedFilesCount + fileEncryptionErrorCount);
                QMessageBox::information(this, "Qccrypt", sReportContent, "&Ok");
            }
            else
                QMessageBox::critical(this, "Qccrypt", "You must enter a password to encrypt this folder!", "&Ok");
        }

    }
}

void CCryptDialog::decryptFolder()
{
    PasswordDialog pwDlg(false, sessionPwd, sessionKeyFile, this);
    if (pwDlg.exec() == QDialog::Accepted)
    {
        QString password = pwDlg.getPassword();
        sessionKeyFile = pwDlg.getKeyFile();
        bool bUseTempFile = pwDlg.isUseTemporaryFileOptionChecked();
        bool bUseKeyFile = pwDlg.isUseKeyFileOptionChecked();

        if (bUseKeyFile)
        {
            // Use a key file
            if (sessionKeyFile.exists())
            {
                successfullyDecryptedFilesCount = 0;
                fileDecryptionErrorCount = 0;
                ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Start of task: decrypt folder \"" + currentFolderToDecryptInfo.path() + "\" ***********");
                ccdlg_ui.progressBar->reset();
                ccdlg_ui.progressBar->setMinimum(0);
                ccdlg_ui.progressBar->setMaximum(subFilesCount);
                currentProgress = 0;
                runFolderDecryption(currentFolderToDecryptInfo.path(), password, bUseTempFile, bUseKeyFile);
                ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - End of task: decrypt folder \"" + currentFolderToDecryptInfo.path() + "\" ***********");
                ccdlg_ui.progressList->scrollToBottom();
                QMessageBox::information(this, "Qccrypt", "Decryption finished.\n\nFiles successfully decrypted: " + tr("%1").arg(successfullyDecryptedFilesCount) + "\nFiles failed to decrypt: "  + tr("%1").arg(fileDecryptionErrorCount) + "\n\nTotal of files processed: " + tr("%1").arg(successfullyDecryptedFilesCount + fileDecryptionErrorCount), "&Ok");
            }
            else
                QMessageBox::critical(this, "Qccrypt", "The key file was not found!\n\n" + sessionKeyFile.filePath(), "&Ok");
        }
        else
        {
            // Do not use a key file
            if (!password.isEmpty())
            {
                sessionPwd = password;
                successfullyDecryptedFilesCount = 0;
                fileDecryptionErrorCount = 0;
                ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Start of task: decrypt folder \"" + currentFolderToDecryptInfo.path() + "\" ***********");
                ccdlg_ui.progressBar->reset();
                ccdlg_ui.progressBar->setMinimum(0);
                ccdlg_ui.progressBar->setMaximum(subFilesCount);
                currentProgress = 0;
                runFolderDecryption(currentFolderToDecryptInfo.path(), password, bUseTempFile, bUseKeyFile);
                ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - End of task: decrypt folder \"" + currentFolderToDecryptInfo.path() + "\" ***********");
                ccdlg_ui.progressList->scrollToBottom();
                QMessageBox::information(this, "Qccrypt", "Decryption finished.\n\nFiles successfully decrypted: " + tr("%1").arg(successfullyDecryptedFilesCount) + "\nFiles failed to decrypt: "  + tr("%1").arg(fileDecryptionErrorCount) + "\n\nTotal of files processed: " + tr("%1").arg(successfullyDecryptedFilesCount + fileDecryptionErrorCount), "&Ok");
            }
            else
                QMessageBox::critical(this, "Qccrypt", "You must enter a password to decrypt this folder!", "&Ok");

        }
    }
}

void CCryptDialog::selectFolderToEncrypt()
{
    QString folderName = QFileDialog::getExistingDirectory(this,
                                                            tr("Select the folder to encrypt"),
                                                            QString(currentFolderToEncryptInfo.path()),
                                                            QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks);
    currentFolderToEncryptInfo.setPath(folderName);
    updateFolderToEncryptProperties();
}

void CCryptDialog::selectFolderToDecrypt()
{
    QString folderName = QFileDialog::getExistingDirectory(this,
                                                            tr("Select the folder to decrypt"),
                                                            QString(currentFolderToDecryptInfo.path()),
                                                            QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks);
    currentFolderToDecryptInfo.setPath(folderName);
    updateFolderToDecryptProperties();
}

void CCryptDialog::clearProgressList()
{
    currentProgress = 0;
    ccdlg_ui.progressBar->setValue(currentProgress);
    ccdlg_ui.progressList->clear();
}

void CCryptDialog::exitQccrypt()
{
    QApplication::exit();
}

void CCryptDialog::dropEvent(QDropEvent *ev)
{
    QList<QUrl> urls = ev->mimeData()->urls();
    if (urls.count() > 1)
    {
        QMessageBox::critical(this, "Qccrypt", "You can drag and drop only one file or folder at the same time!", "&Ok");
    }
    else
    {
        foreach(QUrl url, urls)
        {
            determineAction(QString(url.toLocalFile()));
        }
     }
}

void CCryptDialog::dragEnterEvent(QDragEnterEvent *ev)
{
    ev->accept();
}

void CCryptDialog::determineAction(const QString &fileName)
{
    //Determine the action when a drag and drop has been detected
    QFileInfo droppedFileOrFolder;
    QFileInfo droppedFile;
    QDir droppedFolder;
    droppedFileOrFolder.setFile(fileName);

    if (droppedFileOrFolder.exists())
    {
        if (droppedFileOrFolder.isFile())
        {
            droppedFile = droppedFileOrFolder;
            if (droppedFile.suffix() == "cpt")
            {
                currentFileToDecryptInfo = droppedFile;
                updateFileToDecryptProperties();
                ccdlg_ui.decryptFileRadioButton->click();
            }
            else
            {
                currentFileToEncryptInfo = droppedFile;
                updateFileToEncryptProperties();
                ccdlg_ui.encryptFileRadioButton->click();
            }
        }
        if (droppedFileOrFolder.isDir())
        {
            droppedFolder.setPath(fileName);

            QMessageBox msgBox;
            msgBox.setWindowTitle("Qccrypt");
            msgBox.setWindowFlags(Qt::WindowStaysOnTopHint);
            msgBox.setText(tr("Do you want to encrypt or decrypt this folder?\n\n") + QString(droppedFolder.path()));
            QAbstractButton* pButtonEncrypt = msgBox.addButton(tr("Encrypt"), QMessageBox::YesRole);
            QAbstractButton* pButtonDecrypt = msgBox.addButton(tr("Decrypt"), QMessageBox::NoRole);
            msgBox.addButton(tr("Cancel"), QMessageBox::RejectRole);
            msgBox.exec();

            if(msgBox.clickedButton() == pButtonEncrypt)
            {
                currentFolderToEncryptInfo = droppedFolder;
                updateFolderToEncryptProperties();
                ccdlg_ui.encryptFolderRadioButton->click();
                ccdlg_ui.encryptFolderButton->click();
            }
            else if(msgBox.clickedButton() == pButtonDecrypt)
            {
                currentFolderToDecryptInfo = droppedFolder;
                updateFolderToDecryptProperties();
                ccdlg_ui.decryptFolderRadioButton->click();
                ccdlg_ui.decryptFolderButton->click();
            }
            else {}
        }
    }
    else
        QMessageBox::critical(this, "Qccrypt", "The file or folder specified was not found!\n\n" + fileName, "&Ok");
}

void CCryptDialog::clickStayOnTopCheckbox()
{
    Qt::WindowFlags flags = windowFlags();
    if (ccdlg_ui.stayOnTopCheckBox->isChecked())
    {
        flags |= Qt::WindowStaysOnTopHint;
        settings.setValue("Interface/StayOnTop", true);
    }
    else
    {
        flags  &= ~Qt::WindowStaysOnTopHint;
        settings.setValue("Interface/StayOnTop", false);
    }
    setWindowFlags(flags);
    show();
}

void CCryptDialog::clickKeepExistingFileDuringFileEncryptionCheckBox()
{
    QString settingName = "Encryption/KeepExistingFileDuringFileEncryption";
    if (ccdlg_ui.keepExistingFileCheckBox->isChecked())
        settings.setValue(settingName, true);
    else
        settings.setValue(settingName, false);
}

void CCryptDialog::clickDoNotEncryptAlreadyEncryptedFilesDuringFolderEncryptionCheckBox()
{
    QString settingName = "Encryption/DoNotEncryptAlreadyEncryptedFilesDuringFolderEncryption";
    if (ccdlg_ui.encryptFolderDoNotEncryptAlreadyEncryptedFilesCheckBox->isChecked())
        settings.setValue(settingName, true);
    else
        settings.setValue(settingName, false);
}

void CCryptDialog::clickKeepExistingEncryptedFileDuringFileDecryptionCheckBox()
{
    QString settingName = "Decryption/KeepExistingEncryptedFileDuringFileDecryption";
    if (ccdlg_ui.keepExistingEncryptedFileCheckBox->isChecked())
        settings.setValue(settingName, true);
    else
        settings.setValue(settingName, false);
}

void CCryptDialog::clickTabQccryptMode(int tabIndex)
{
    if (tabIndex == 0)
    //Encryption mode tab
    {
        if (ccdlg_ui.encryptFileRadioButton->isChecked())
            showEncryptFileGroupBox();
        else
            showEncryptFolderGroupBox();
    }
    else
    //Decryption mode tab
    {
        if (ccdlg_ui.decryptFileRadioButton->isChecked())
            showDecryptFileGroupBox();
        else
            showDecryptFolderGroupBox();
    }
}

QString CCryptDialog::getRandomString(int length)
{

    srand(QDateTime::currentMSecsSinceEpoch());

    const char ch[] = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
    int size = sizeof(ch);

    char* str = new char[length + 1];

    int num = 0;
    for (int i = 0; i < length; ++i)
    {

        num = rand() % (size - 1);
        str[i] = ch[num];
    }

    QString res(str);
    return res;
}
