/*
* This file is part of Qccrypt, an open-source cross-platform frontend for ccrypt
* encryption tool.
* Copyright (C) 2006-2014  Philippe Beaureilles
* This source code is based on a part of QPaMaT (password managing tool) source code
* developed by Bernhard Walle (<bernhard.walle@gmx.de>), and adapted to my needs, as it is
* permitted by the GNU GPL license.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
* Contact e-mail: Philippe Beaureilles <qccrypt@free.fr>
* Program URL   : http://qccrypt.free.fr/
*
*/

#include <QDialog>
#include <QLineEdit>
#include <QLayout>
#include <QLabel>
#include <QMessageBox>
#include <QPushButton>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QGridLayout>

#include "newpassworddialog.h"

NewPasswordDialog::NewPasswordDialog(QWidget* parent)
    : QDialog(parent)
{
    this->setWindowTitle("Qccrypt");
    
    createAndLayout();
    
    // validators
    PasswordValidator* validator = new PasswordValidator(this);
    m_firstPasswordEdit->setValidator(validator);
    m_secondPasswordEdit->setValidator(validator);
    
    // communication
    connect(m_okButton, SIGNAL(clicked()), SLOT(accept()));
    connect(m_cancelButton, SIGNAL(clicked()), SLOT(reject()));
    connect(m_firstPasswordEdit, SIGNAL(textChanged(const QString&)), SLOT(checkOkEnabled()));
    connect(m_secondPasswordEdit, SIGNAL(textChanged(const QString&)), SLOT(checkOkEnabled()));
    
    connect(m_firstPasswordEdit, SIGNAL(gotFocus()), SLOT(grabFirstPassword()));
    connect(m_firstPasswordEdit, SIGNAL(lostFocus()), SLOT(release()));
    connect(m_secondPasswordEdit, SIGNAL(gotFocus()), SLOT(grabSecondPassword()));
    connect(m_secondPasswordEdit, SIGNAL(lostFocus()), SLOT(release()));
}

/*!
    Creates and layout the widgets of the dialog.
*/
void NewPasswordDialog::createAndLayout()
{
    m_firstPasswordEdit = new QLineEdit(this);
    m_firstPasswordEdit->setEchoMode(QLineEdit::Password);
    m_firstPasswordEdit->setMinimumWidth(250);
    
    m_secondPasswordEdit = new QLineEdit(this);
    m_secondPasswordEdit->setEchoMode(QLineEdit::Password);
    m_secondPasswordEdit->setMinimumWidth(250);
    
    const QString newText = tr("Enter the <b>password</b> two times for verification. The "
        "minimum length<br> must be one character, but for security reasons at least six<br> "
        "characters is better!");
    
    // labels
    QLabel* label = new QLabel(newText, this);
    QLabel* first = new QLabel(tr("&Password:"), this);
    first->setBuddy(m_firstPasswordEdit);
    QLabel* second = new QLabel(tr("&Verification:"), this);
    second->setBuddy(m_secondPasswordEdit);
    
    // buttons
    m_okButton = new QPushButton(tr("OK"), this);
    m_okButton->setDefault(true);
    m_okButton->setEnabled(false);
    m_cancelButton = new QPushButton(tr("Cancel"), this);
    
    // create layouts
    QVBoxLayout* layout = new QVBoxLayout(this);
    QGridLayout* textfields = new QGridLayout(0);
    QHBoxLayout* buttonLayout = new QHBoxLayout(0);
    buttonLayout->setSpacing(6);
    
    // layout elements
    buttonLayout->addStretch(10);
    buttonLayout->addWidget(m_okButton);
    buttonLayout->addWidget(m_cancelButton);
    
    int i = 0;
    textfields->addWidget(first, i+0, 0);
    textfields->addWidget(m_firstPasswordEdit, i+0, 1);
    textfields->addWidget(second, i+1, 0);
    textfields->addWidget(m_secondPasswordEdit, i+1, 1);
    
    layout->addWidget(label);
    layout->addLayout(textfields);
    layout->addSpacing(7);
    layout->addLayout(buttonLayout);
}

/*!
    Method that is called if the user presses Ok.
    It checks the password for quality and if the old password was right. If not, it displays
    a message box and the user has the chance to try again.
*/
void NewPasswordDialog::accept()
{
    if (m_firstPasswordEdit->text()!= m_secondPasswordEdit->text())
    {
        QMessageBox::warning(this, "Qccrypt",
               "<qt>"+tr("The confirmation password was not the same as the first one. "
               "Re-enter the confirmation or both.")+"</qt>",
               QMessageBox::Ok | QMessageBox::Default, QMessageBox::NoButton);
        return;
    }
    QDialog::accept();
}

/*!
    Grabs the first password edit. Should be connected to the gotFocus() signal of the password
    lineedit widget.
*/
void NewPasswordDialog::grabFirstPassword()
{
    m_firstPasswordEdit->grabKeyboard();
}


/*!
    Grabs the second password edit. Should be connected to the gotFocus() signal of the password
    lineedit widget.
*/
void NewPasswordDialog::grabSecondPassword()
{
    m_secondPasswordEdit->grabKeyboard();
}


/*!
    Releases the password edit. Should be connected to the lostFocus() signal of the password
    lineedit widget
*/
void NewPasswordDialog::release()
{
    QWidget* widget = keyboardGrabber();
    if (widget)
    {
        widget->releaseKeyboard();
    }
}

/*!
    Returns the (new) password the user has entered.
    \return the password
*/
QString NewPasswordDialog::getPassword() const
{
    return m_firstPasswordEdit->text();
}


/*!
    Checks the user's input and enables the Ok button if necessary. This slot is called always
    if the users changes the input.
*/
void NewPasswordDialog::checkOkEnabled() const
{
    if (m_firstPasswordEdit->hasAcceptableInput() && m_secondPasswordEdit->hasAcceptableInput()
            && m_firstPasswordEdit->text().length() == m_secondPasswordEdit->text().length())
    {
        m_okButton->setEnabled(true);
    }
    else
    {
        m_okButton->setEnabled(false);
    } 
}

PasswordValidator::PasswordValidator(QObject* parent)
    : QValidator(parent)
{}


/*!
    Does the validation. It returns QValidator::Intermediate if the password has less than
    one characters and QValidator::Acceptable if it has equal or more than one.
    \param input the input
    \param pos the position of the inserted stuff (unused here)
*/
PasswordValidator::State PasswordValidator::validate(QString& input, int& pos) const
{
    if (input.length() < 1)
    {
        return QValidator::Intermediate;
    }
    else
    {
        return QValidator::Acceptable;
    }
}


