/*
* This file is part of Qccrypt, an open-source cross-platform frontend for ccrypt
* encryption tool.
* Copyright (C) 2006  Philippe Beaureilles
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
* Contact e-mail: Philippe Beaureilles <qccrypt@free.fr>
* Program URL   : http://qccrypt.free.fr/
*
*/

#include <QtGui>
#include <QProcess>
#include <QTabWidget>
#include <QCursor>

#include "ccdlg.h"
#include "newpassworddialog.h"

#define round_up(n, k) (((n) - 1 + (k)) / (k)) //Thanks to Sebastian Lisken for this tip

CCryptDialog::CCryptDialog(const QString &fileName, QWidget *parent)
    : QDialog(parent)
{
	ccdlg_ui.setupUi(this);
	
	connect(ccdlg_ui.encryptOpenFileNameButton, SIGNAL(clicked()),
			this, SLOT(setEncryptOpenFileName()));
	connect(ccdlg_ui.encryptFileButton, SIGNAL(clicked()),
			this, SLOT(encryptFile()));
	connect(ccdlg_ui.decryptOpenFileNameButton, SIGNAL(clicked()),
			this, SLOT(setDecryptOpenFileName()));
	connect(ccdlg_ui.decryptFileButton, SIGNAL(clicked()),
			this, SLOT(decryptFile()));
	connect(ccdlg_ui.aboutButton, SIGNAL(clicked()),
			this, SLOT(about()));
	connect(ccdlg_ui.encryptFileRadioButton, SIGNAL(clicked()),
			this, SLOT(showEncryptFileGroupBox()));
	connect(ccdlg_ui.encryptFolderRadioButton, SIGNAL(clicked()),
			this, SLOT(showEncryptFolderGroupBox()));
	connect(ccdlg_ui.decryptFileRadioButton, SIGNAL(clicked()),
			this, SLOT(showDecryptFileGroupBox()));
	connect(ccdlg_ui.decryptFolderRadioButton, SIGNAL(clicked()),
			this, SLOT(showDecryptFolderGroupBox()));
	connect(ccdlg_ui.encryptFolderButton, SIGNAL(clicked()),
			this, SLOT(encryptFolder()));
	connect(ccdlg_ui.decryptFolderButton, SIGNAL(clicked()),
			this, SLOT(decryptFolder()));
	connect(ccdlg_ui.encryptOpenFolderNameButton, SIGNAL(clicked()),
			this, SLOT(setEncryptOpenFolderName()));
	connect(ccdlg_ui.decryptOpenFolderNameButton, SIGNAL(clicked()),
			this, SLOT(setDecryptOpenFolderName()));
        connect(ccdlg_ui.clearProgressListButton, SIGNAL(clicked()),
                        this, SLOT(clearProgressList()));
        connect(ccdlg_ui.exitButton, SIGNAL(clicked()),
                        this, SLOT(exitQccrypt()));


	showEncryptFileGroupBox();
	showDecryptFileGroupBox();
	emptyAllFields();
	
	if (!fileName.isEmpty())
	{
		currentFileInfo.setFile(fileName);
		if	(currentFileInfo.isDir())
		{
			if (currentFolderInfo.exists())
			{
				currentFolderInfo.setPath(fileName);
				ccdlg_ui.decryptFolderRadioButton->click();
				updateFolderToEncryptProperties();
				ccdlg_ui.encryptFolderRadioButton->click();
				ccdlg_ui.decryptFolderNameEdit->setText(currentFolderInfo.path());
				ccdlg_ui.decryptFolderSizeValueLabel->setText(tr("%1 Kbytes").arg(round_up(getFolderSize(currentFolderInfo.path()),1024)));
                                ccdlg_ui.decryptFolderSubFoldersCountValueLabel->setText(tr("%1").arg(getSubFoldersCount(currentFolderInfo.path())));
                                subFilesCount = getSubFilesCount(currentFolderInfo.path());
                                ccdlg_ui.decryptFolderSubFilesCountValueLabel->setText(tr("%1").arg(subFilesCount));
		    	ccdlg_ui.decryptFolderButton->setEnabled ( true );
			}
		}
		else if (currentFileInfo.exists())
		{
			if (currentFileInfo.suffix() == "cpt")
				updateFileToDecryptProperties();
			else
				updateFileToEncryptProperties();
    	}
    	else
    	{
    		QMessageBox::warning(this, "Qccrypt", "The file specified as a parameter on the command line did not found!", "&Ok");
    		emptyAllFields();
   		}
	}
	else
   		emptyAllFields();
}

///////////////////////////////
// Display functions
///////////////////////////////
void CCryptDialog::updateFileToEncryptProperties()
{
	if (currentFileInfo.exists())
	{
            ccdlg_ui.tabQccryptMode->setCurrentWidget(ccdlg_ui.encryptionTab);
            ccdlg_ui.encryptFileNameEdit->setText(currentFileInfo.fileName());
            ccdlg_ui.encryptFilePathValueLabel->setText(currentFileInfo.path());
            qlonglong size = round_up(currentFileInfo.size(), 1024);
            ccdlg_ui.encryptFileSizeValueLabel->setText(tr("%1 K").arg(size));
            ccdlg_ui.encryptFileLastReadValueLabel->setText(currentFileInfo.lastRead().toString());
	    ccdlg_ui.encryptFileLastModValueLabel->setText(currentFileInfo.lastModified().toString());
	    ccdlg_ui.encryptFileButton->setEnabled ( true );
	    ccdlg_ui.keepExistingFileCheckBox->setEnabled ( true );
	    emptyDecryptFileFields();
   	}
   	else
   	{
   		emptyEncryptFileFields();
   		emptyDecryptFileFields();
  	}
}

void CCryptDialog::updateFileToDecryptProperties()
{
	if (currentFileInfo.exists())
	{
		ccdlg_ui.tabQccryptMode->setCurrentWidget(ccdlg_ui.decryptionTab);
		ccdlg_ui.decryptFileNameEdit->setText(currentFileInfo.fileName());
		ccdlg_ui.decryptFilePathValueLabel->setText(currentFileInfo.path());
		qlonglong size = round_up(currentFileInfo.size(), 1024);
		ccdlg_ui.decryptFileSizeValueLabel->setText(tr("%1 K").arg(size));
		ccdlg_ui.decryptFileLastReadValueLabel->setText(currentFileInfo.lastRead().toString());
	    ccdlg_ui.decryptFileLastModValueLabel->setText(currentFileInfo.lastModified().toString());
	    ccdlg_ui.decryptFileButton->setEnabled ( true );
	    ccdlg_ui.keepExistingEncryptedFileCheckBox->setEnabled ( true );
	    emptyEncryptFileFields();
   	}
   	else
   	{
   		emptyEncryptFileFields();
   		emptyDecryptFileFields();
  	}
}

void CCryptDialog::updateFolderToEncryptProperties()
{
	if (currentFolderInfo.exists())
	{
		ccdlg_ui.tabQccryptMode->setCurrentWidget(ccdlg_ui.encryptionTab);
		ccdlg_ui.encryptFolderNameEdit->setText(currentFolderInfo.path());
		ccdlg_ui.encryptFolderSizeValueLabel->setText(tr("%1 Kbytes").arg(round_up(getFolderSize(currentFolderInfo.path()),1024)));
                ccdlg_ui.encryptFolderSubFoldersCountValueLabel->setText(tr("%1").arg(getSubFoldersCount(currentFolderInfo.path())));
                subFilesCount = getSubFilesCount(currentFolderInfo.path());
                ccdlg_ui.encryptFolderSubFilesCountValueLabel->setText(tr("%1").arg(subFilesCount));
                ccdlg_ui.encryptFolderButton->setEnabled ( true );
                emptyDecryptFolderFields();
	}
	else
	{
   		emptyEncryptFolderFields();
   		emptyDecryptFolderFields();
	}
}

void CCryptDialog::updateFolderToDecryptProperties()
{
	if (currentFolderInfo.exists())
	{
		ccdlg_ui.tabQccryptMode->setCurrentWidget(ccdlg_ui.decryptionTab);
		ccdlg_ui.decryptFolderNameEdit->setText(currentFolderInfo.path());
		ccdlg_ui.decryptFolderSizeValueLabel->setText(tr("%1 Kbytes").arg(round_up(getFolderSize(currentFolderInfo.path()),1024)));
                ccdlg_ui.decryptFolderSubFoldersCountValueLabel->setText(tr("%1").arg(getSubFoldersCount(currentFolderInfo.path())));
                subFilesCount = getSubFilesCount(currentFolderInfo.path());
                ccdlg_ui.decryptFolderSubFilesCountValueLabel->setText(tr("%1").arg(subFilesCount));
                ccdlg_ui.decryptFolderButton->setEnabled ( true );
                emptyEncryptFolderFields();
	}
	else
	{
   		emptyEncryptFolderFields();
   		emptyDecryptFolderFields();
	}
}

void CCryptDialog::emptyEncryptFileFields()
{
	ccdlg_ui.encryptFileNameEdit->setText("");
	ccdlg_ui.encryptFilePathValueLabel->setText("");
	ccdlg_ui.encryptFileSizeValueLabel->setText("");
	ccdlg_ui.encryptFileLastReadValueLabel->setText("");
        ccdlg_ui.encryptFileLastModValueLabel->setText("");
	ccdlg_ui.encryptFileButton->setEnabled ( false );
        ccdlg_ui.keepExistingFileCheckBox->setEnabled ( false );
}

void CCryptDialog::emptyDecryptFileFields()
{
	ccdlg_ui.decryptFileNameEdit->setText("");
	ccdlg_ui.decryptFilePathValueLabel->setText("");
	ccdlg_ui.decryptFileSizeValueLabel->setText("");
	ccdlg_ui.decryptFileLastReadValueLabel->setText("");
    ccdlg_ui.decryptFileLastModValueLabel->setText("");
	ccdlg_ui.decryptFileButton->setEnabled ( false );
    ccdlg_ui.keepExistingEncryptedFileCheckBox->setEnabled ( false );
}

void CCryptDialog::emptyEncryptFolderFields()
{
	ccdlg_ui.encryptFolderNameEdit->setText("");
	ccdlg_ui.encryptFolderSizeValueLabel->setText("");
	ccdlg_ui.encryptFolderSubFoldersCountValueLabel->setText("");
    ccdlg_ui.encryptFolderSubFilesCountValueLabel->setText("");
	ccdlg_ui.encryptFolderButton->setEnabled ( false );
}

void CCryptDialog::emptyDecryptFolderFields()
{
	ccdlg_ui.decryptFolderNameEdit->setText("");
	ccdlg_ui.decryptFolderSizeValueLabel->setText("");
	ccdlg_ui.decryptFolderSubFoldersCountValueLabel->setText("");
        ccdlg_ui.decryptFolderSubFilesCountValueLabel->setText("");
	ccdlg_ui.decryptFolderButton->setEnabled ( false );
}

void CCryptDialog::emptyAllFields()
{
	emptyEncryptFileFields();
	emptyEncryptFolderFields();
	emptyDecryptFileFields();
	emptyDecryptFolderFields();
}

void CCryptDialog::showEncryptFileGroupBox()
{
	ccdlg_ui.encryptFolderGroupBox->hide();
	ccdlg_ui.encryptFileGroupBox->show();
        ccdlg_ui.encryptFileInformationGroupBox->show();
        ccdlg_ui.encryptFolderInformationGroupBox->hide();
}

void CCryptDialog::showDecryptFileGroupBox()
{
	ccdlg_ui.decryptFolderGroupBox->hide();
	ccdlg_ui.decryptFileGroupBox->show();
        ccdlg_ui.decryptFileInformationGroupBox->show();
        ccdlg_ui.decryptFolderInformationGroupBox->hide();
}

void CCryptDialog::showEncryptFolderGroupBox()
{
	ccdlg_ui.encryptFileGroupBox->hide();
	ccdlg_ui.encryptFolderGroupBox->show();
        ccdlg_ui.encryptFileInformationGroupBox->hide();
        ccdlg_ui.encryptFolderInformationGroupBox->show();
}

void CCryptDialog::showDecryptFolderGroupBox()
{
	ccdlg_ui.decryptFileGroupBox->hide();
	ccdlg_ui.decryptFolderGroupBox->show();
        ccdlg_ui.decryptFileInformationGroupBox->hide();
        ccdlg_ui.decryptFolderInformationGroupBox->show();
}

qlonglong CCryptDialog::getFolderSize(QString path)
{
	qlonglong size=0;
	QDir folder;
	folder.setPath(path);
	QFileInfoList fileInfoList = folder.entryInfoList(QDir::AllEntries|QDir::NoDotAndDotDot);
	for(int i = 0; i < fileInfoList.size(); ++i)
	{
		if(fileInfoList.at(i).isDir())
			size+=getFolderSize(fileInfoList.at(i).path() + "/" + fileInfoList.at(i).fileName());
		else
	  		size+=fileInfoList.at(i).size();
	}
	return size;
}

qlonglong CCryptDialog::getSubFoldersCount(QString path)
{
	qlonglong count=0;
	QDir folder;
	folder.setPath(path);
	QFileInfoList fileInfoList = folder.entryInfoList(QDir::AllEntries|QDir::NoDotAndDotDot);
	for(int i = 0; i < fileInfoList.size(); ++i)
	{
		if(fileInfoList.at(i).isDir())
		{
			count++;
                        count+=getSubFoldersCount(fileInfoList.at(i).path() + "/" + fileInfoList.at(i).fileName());
		}
	}
	return count;
}

qlonglong CCryptDialog::getSubFilesCount(QString path)
{
	qlonglong count=0;
	QDir folder;
	folder.setPath(path);
	QFileInfoList fileInfoList = folder.entryInfoList(QDir::AllEntries|QDir::NoDotAndDotDot);
	for(int i = 0; i < fileInfoList.size(); ++i)
	{
		if (fileInfoList.at(i).isFile())
			count++;
		if (fileInfoList.at(i).isDir())
                        count+=getSubFilesCount(fileInfoList.at(i).path() + "/" + fileInfoList.at(i).fileName());
	}
	return count;
}

///////////////////////////////
// Connectors
///////////////////////////////
void CCryptDialog::setEncryptOpenFileName()
{
	QString fileName = QFileDialog::getOpenFileName(this,
								tr("Select the file to encrypt"),
								QString(currentFileInfo.absoluteFilePath()),
								tr("All Files (*)"));

	currentFileInfo.setFile(fileName);
	updateFileToEncryptProperties();
}

void CCryptDialog::encryptFile()
{
	NewPasswordDialog pwDlg(this);
	if (pwDlg.exec() == QDialog::Accepted)
	{
		QString password = pwDlg.getPassword();
		
		if (!password.isEmpty())
		{
			QString targetFileName;
			targetFileName = QString(currentFileInfo.absoluteFilePath());
			QFileInfo targetFile;
                        targetFile = QFileInfo(targetFileName  + ".cpt");

			if (targetFile.isFile())
			{
				int reply = QMessageBox::warning(this, tr("Replace file confirmation"),
								tr("The target file already exists. Do you want to overwrite it ?"),
								tr("&Yes"),
								tr("&No"));
				if (reply == 1)
					return;
			}

                        ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Start of task: encrypt file \"" + targetFileName + "\" ***********");
                        ccdlg_ui.progressBar->reset();
                        ccdlg_ui.progressBar->setMinimum(0);
                        ccdlg_ui.progressBar->setMaximum(1);
                        currentProgress = 0;
                        ccdlg_ui.progressBar->setValue(currentProgress);
                        int resultCode = runFileEncryption(targetFileName, password);
                        ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Start of task: encrypt file \"" + targetFileName + "\" ***********");
                        ccdlg_ui.progressList->scrollToBottom();
                        currentProgress++;
                        ccdlg_ui.progressBar->setValue(currentProgress);

                        switch (resultCode) {
				case 0 :
					QMessageBox::information(this, "Qccrypt", "Encryption successful.", "&Ok");
					emptyAllFields();
					break;
				default :
					QMessageBox::critical(this, "Qccrypt", "Encryption error.", "&Ok");
			}
		}
		else
			QMessageBox::warning(this, "Qccrypt", "You must enter a password to encrypt this file!", "&Ok");
	}
}


void CCryptDialog::setDecryptOpenFileName()
{
	QString fileName = QFileDialog::getOpenFileName(this,
								tr("Select the file to decrypt"),
								QString(currentFileInfo.absoluteFilePath()),
								tr("ccrypt crypted file (*.cpt)"));
	currentFileInfo.setFile(fileName);
	updateFileToDecryptProperties();
}

void CCryptDialog::decryptFile()
{
	bool ok;
	QString password = QInputDialog::getText(this, tr("Password"),
										tr("Enter the password:"), QLineEdit::Password,
										"", &ok);
	if (ok)
	{
		if (!password.isEmpty())
		{
			QString targetFileName;
			targetFileName = QString(currentFileInfo.absoluteFilePath());
			targetFileName = targetFileName.left(targetFileName.length() - 4);
			QFileInfo targetFile;
			targetFile = QFileInfo(targetFileName);

			if (targetFile.isFile())
			{
				int reply = QMessageBox::warning(this, tr("Replace file confirmation"),
								tr("The target file already exists. Do you want to overwrite it ?"),
								tr("&Yes"),
								tr("&No"));
				if (reply == 1)
					return;
			}

                        ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Start of task: decrypt file \"" + targetFileName + "\" ***********");
                        ccdlg_ui.progressBar->reset();
                        ccdlg_ui.progressBar->setMinimum(0);
                        ccdlg_ui.progressBar->setMaximum(1);
                        currentProgress = 0;
                        ccdlg_ui.progressBar->setValue(currentProgress);
                        int resultCode = runFileDecryption(targetFileName, password);
                        ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Start of task: decrypt file \"" + targetFileName + "\" ***********");
                        ccdlg_ui.progressList->scrollToBottom();
                        currentProgress++;
                        ccdlg_ui.progressBar->setValue(currentProgress);

                        switch (resultCode) {
                                case 0 :
					QMessageBox::information(this, "Qccrypt", "Decryption successful.", "&Ok");
					emptyAllFields();
					break;
				case 4 :
					QMessageBox::critical(this, "Qccrypt", "The password specified was incorrect!", "&Ok");
					break;
				default :
					QMessageBox::critical(this, "Qccrypt", "Decryption error.", "&Ok");
			}
		}
		else
			QMessageBox::warning(this, "Qccrypt", "You must enter the password to decrypt this file!", "&Ok");
	}
}

void CCryptDialog::about()
{
        QMessageBox::information(this, "About Qccrypt 0.3.1", "Qccrypt is an Open Source frontend for ccrypt encryption/decryption tool.\nIt is written in C++ with QT4 libraries by Philippe Beaureilles.\n\nThis program is free software. You can redistribute it and/or modify it under the terms of\nthe GNU General Public License as published by the Free Software Foundation\neither version 2 of the License, or (at your option) any later version.\n\nSpecial thanks to Sebastian Lisken for his contribution to Qccrypt (errors report, corrections of language mistakes, portions of code and suggestions).\n\n http://qccrypt.free/", "&Ok");
}

int CCryptDialog::runFileEncryption(QString fileName, QString password)
{
    QString program = "ccrypt";
    QStringList arguments;
    QProcess *encryptProcess = new QProcess(this);
    QFileInfo fileInfo;
    fileInfo = QFileInfo(fileName);

    QListWidgetItem *newItem = new QListWidgetItem;
    newItem->setText(QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Encryting \"" + fileName + "\"...");
    ccdlg_ui.progressList->addItem(newItem);
    ccdlg_ui.progressList->scrollToBottom();

    if (ccdlg_ui.keepExistingFileCheckBox->isChecked())
    {
            arguments << "-e" << "-f" << "-K" << password;
            encryptProcess->setStandardInputFile(fileName);
            encryptProcess->setStandardOutputFile(fileName + ".cpt");
    }
    else
            arguments << "-e" << "-f" << "-K" << password << fileName ;
    QApplication::setOverrideCursor(Qt::WaitCursor);
    encryptProcess->start(program, arguments);
    QApplication::processEvents();
    encryptProcess->waitForFinished(-1);
    QApplication::restoreOverrideCursor();

    switch (encryptProcess->exitCode ()) {
            case 0 :
                    newItem->setText(newItem->text() + " OK!");
                    emptyAllFields();
                    break;
            default :
                    newItem->setText(newItem->text() + " Failed!");
    }

    return encryptProcess->exitCode ();
}

int CCryptDialog::runFolderEncryption(QString folderName, QString password)
{
    QDir folderDir;
    folderDir.setPath(folderName);
    QFileInfoList fileInfoList = folderDir.entryInfoList(QDir::AllEntries|QDir::NoDotAndDotDot);
    for(int i = 0; i < fileInfoList.size(); ++i)
    {
        if (fileInfoList.at(i).isFile())
        {
            runFileEncryption(fileInfoList.at(i).absoluteFilePath(), password);
            currentProgress++;
        }
        if (fileInfoList.at(i).isDir())
        {
            runFolderEncryption(fileInfoList.at(i).absoluteFilePath(), password);
        }
        ccdlg_ui.progressBar->setValue(currentProgress);
    }
    return 0;
}

int CCryptDialog::runFolderDecryption(QString folderName, QString password)
{
    QDir folderDir;
    folderDir.setPath(folderName);
    QFileInfoList fileInfoList = folderDir.entryInfoList(QDir::AllEntries|QDir::NoDotAndDotDot);
    for(int i = 0; i < fileInfoList.size(); ++i)
    {
        if (fileInfoList.at(i).isFile())
        {
            runFileDecryption(fileInfoList.at(i).absoluteFilePath(), password);
            currentProgress++;
        }
        if (fileInfoList.at(i).isDir())
        {
            runFolderDecryption(fileInfoList.at(i).absoluteFilePath(), password);
        }
        ccdlg_ui.progressBar->setValue(currentProgress);
    }
    return 0;
}


int CCryptDialog::runFileDecryption(QString fileName, QString password)
{
    QString program = "ccrypt";
    QStringList arguments;
    QProcess *encryptProcess = new QProcess(this);

    QListWidgetItem *newItem = new QListWidgetItem;
    newItem->setText(QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Decryting \"" + fileName + "\"...");
    ccdlg_ui.progressList->addItem(newItem);
    ccdlg_ui.progressList->scrollToBottom();

    if (ccdlg_ui.keepExistingEncryptedFileCheckBox->isChecked())
    {
        arguments << "-d" << "-f" << "-K" << password;
        encryptProcess->setStandardInputFile(fileName);
        fileName.truncate(fileName.length() - 4);
        encryptProcess->setStandardOutputFile(fileName);
    }
    else
        arguments << "-d" << "-f" << "-K" << password << fileName ;
    QApplication::setOverrideCursor(Qt::WaitCursor);
    encryptProcess->start(program, arguments);
    QApplication::processEvents();
    encryptProcess->waitForFinished();
    QApplication::restoreOverrideCursor();

    switch (encryptProcess->exitCode ()) {
        case 0 :
            newItem->setText(newItem->text() + " OK!");
            emptyAllFields();
            break;
        default :
            newItem->setText(newItem->text() + " Failed!");
    }

    return encryptProcess->exitCode ();
}

void CCryptDialog::encryptFolder()
{
	NewPasswordDialog pwDlg(this);
	if (pwDlg.exec() == QDialog::Accepted)
	{
		QString password = pwDlg.getPassword();
		
		if (!password.isEmpty())
		{
                    QFileInfoList fileInfoList = currentFolderInfo.entryInfoList(QDir::AllEntries|QDir::NoDotAndDotDot);
                    ccdlg_ui.progressBar->reset();
                    ccdlg_ui.progressBar->setMinimum(0);
                    ccdlg_ui.progressBar->setMaximum(subFilesCount);
                    currentProgress = 0;
                    ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Start of task: encrypt folder \"" + currentFolderInfo.path() + "\" ***********");
                    runFolderEncryption(currentFolderInfo.path(), password);
                    ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - End of task: encrypt folder \"" + currentFolderInfo.path() + "\" ***********");
                    ccdlg_ui.progressList->scrollToBottom();
                    QMessageBox::information(this, "Qccrypt", "Encryption finished.", "&Ok");
                }
		else
			QMessageBox::warning(this, "Qccrypt", "You must enter a password to encrypt this folder!", "&Ok");
	}	
}

void CCryptDialog::decryptFolder()
{
    bool ok;
    QString password = QInputDialog::getText(this, tr("Password"),
                                                                        tr("Enter the password:"), QLineEdit::Password,
                                                                        "", &ok);
    if (ok)
    {
        if (!password.isEmpty())
        {
            ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - Start of task: decrypt folder \"" + currentFolderInfo.path() + "\" ***********");
            QFileInfoList fileInfoList = currentFolderInfo.entryInfoList(QDir::AllEntries|QDir::NoDotAndDotDot);
            ccdlg_ui.progressBar->reset();
            ccdlg_ui.progressBar->setMinimum(0);
            ccdlg_ui.progressBar->setMaximum(subFilesCount);
            currentProgress = 0;
            runFolderDecryption(currentFolderInfo.path(), password);
            ccdlg_ui.progressList->addItem("*********** " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") +  " - End of task: decrypt folder \"" + currentFolderInfo.path() + "\" ***********");
            ccdlg_ui.progressList->scrollToBottom();
            QMessageBox::information(this, "Qccrypt", "Decryption finished.", "&Ok");
        }
        else
                QMessageBox::warning(this, "Qccrypt", "You must enter a password to decrypt this folder!", "&Ok");
    }
}

void CCryptDialog::setEncryptOpenFolderName()
{
    QString folderName = QFileDialog::getExistingDirectory(this,
                                                            tr("Select the folder to encrypt"),
                                                            QString(currentFolderInfo.path()),
                                                            QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks);
    currentFolderInfo.setPath(folderName);
    updateFolderToEncryptProperties();
}

void CCryptDialog::setDecryptOpenFolderName()
{
    QString folderName = QFileDialog::getExistingDirectory(this,
                                                            tr("Select the folder to decrypt"),
                                                            QString(currentFolderInfo.path()),
                                                            QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks);
    currentFolderInfo.setPath(folderName);
    updateFolderToDecryptProperties();
}

void CCryptDialog::clearProgressList()
{
    currentProgress = 0;
    ccdlg_ui.progressBar->setValue(currentProgress);
    ccdlg_ui.progressList->clear();
}

void CCryptDialog::exitQccrypt()
{
    QApplication::exit();
}
