/*
* This file is part of Qccrypt, an open-source cross-platform frontend for ccrypt
* encryption tool.
* Copyright (C) 2006  Philippe Beaureilles
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
* Contact e-mail: Philippe Beaureilles <qccrypt@free.fr>
* Program URL   : http://qccrypt.free.fr/
*
*/

#include <QtGui>
#include <QProcess>
#include <QTabWidget>

#include "ccdlg.h"

CCryptDialog::CCryptDialog(const QString &fileName, QWidget *parent)
    : QDialog(parent)
{
	ccdlg_ui.setupUi(this);
	
	connect(ccdlg_ui.encryptOpenFileNameButton, SIGNAL(clicked()),
			this, SLOT(setEncryptOpenFileName()));

	connect(ccdlg_ui.encryptButton, SIGNAL(clicked()),
			this, SLOT(encryptFile()));

	connect(ccdlg_ui.decryptOpenFileNameButton, SIGNAL(clicked()),
			this, SLOT(setDecryptOpenFileName()));

	connect(ccdlg_ui.decryptButton, SIGNAL(clicked()),
			this, SLOT(decryptFile()));

	connect(ccdlg_ui.aboutButton, SIGNAL(clicked()),
			this, SLOT(about()));
	
	if (!fileName.isEmpty())
	{
		currentFileInfo.setFile(fileName);

		if (currentFileInfo.exists())
		{
			if (currentFileInfo.suffix() == "cpt")
			{
				ccdlg_ui.tabQccryptMode->setCurrentWidget(ccdlg_ui.decryptionTab);

				ccdlg_ui.decryptFileNameEdit->setText(fileName);
		
				ccdlg_ui.decryptPathValueLabel->setText(currentFileInfo.absoluteFilePath());
				
				qlonglong size = currentFileInfo.size()/1024;
				if ((size == 0) and (currentFileInfo.size()) > 0 )
					size = 1;
				ccdlg_ui.decryptSizeValueLabel->setText(tr("%1 K").arg(size));
				
				ccdlg_ui.decryptLastReadValueLabel->setText(currentFileInfo.lastRead().toString());
		
			    ccdlg_ui.decryptLastModValueLabel->setText(currentFileInfo.lastModified().toString());
			    
			    ccdlg_ui.decryptButton->setEnabled ( true );
			}
			else
			{
				ccdlg_ui.tabQccryptMode->setCurrentWidget(ccdlg_ui.encryptionTab);

				ccdlg_ui.encryptFileNameEdit->setText(fileName);
		
				ccdlg_ui.encryptPathValueLabel->setText(currentFileInfo.absoluteFilePath());
				
				qlonglong size = currentFileInfo.size()/1024;
				if ((size == 0) and (currentFileInfo.size()) > 0 )
					size = 1;
				ccdlg_ui.encryptSizeValueLabel->setText(tr("%1 K").arg(size));
				
				ccdlg_ui.encryptLastReadValueLabel->setText(currentFileInfo.lastRead().toString());
		
			    ccdlg_ui.encryptLastModValueLabel->setText(currentFileInfo.lastModified().toString());
			    
			    ccdlg_ui.encryptButton->setEnabled ( true );
	    	}
    	}
    	else
    	{
    		QMessageBox::warning(this, "Qccrypt", "The file specified as a parameter on the command line did not found !", "&Ok");
    		emptyEncryptFileFields();
    		emptyDecryptFileFields();
   		}
	}
	else
	{
   		emptyEncryptFileFields();
   		emptyDecryptFileFields();
	}
}

void CCryptDialog::setEncryptOpenFileName()
{
	QString fileName = QFileDialog::getOpenFileName(this,
								tr("Select a file to encrypt"),
								ccdlg_ui.encryptFileNameEdit->text(),
								tr("All Files (*)"));
	
	if (!fileName.isEmpty())
	{
		currentFileInfo.setFile(fileName);

		if (currentFileInfo.exists())
		{
			currentFileInfo.setFile(fileName);
			ccdlg_ui.encryptFileNameEdit->setText(fileName);
		
			ccdlg_ui.encryptPathValueLabel->setText(currentFileInfo.absoluteFilePath());
			
			qlonglong size = currentFileInfo.size()/1024;
			if ((size == 0) and (currentFileInfo.size()) > 0 )
				size = 1;
			ccdlg_ui.encryptSizeValueLabel->setText(tr("%1 K").arg(size));
			
			ccdlg_ui.encryptLastReadValueLabel->setText(currentFileInfo.lastRead().toString());
	
		    ccdlg_ui.encryptLastModValueLabel->setText(currentFileInfo.lastModified().toString());
		    
	    	ccdlg_ui.encryptButton->setEnabled ( true );
   		}
   		else
   		{
   			emptyEncryptFileFields();
   			emptyDecryptFileFields();
  		}
	}
	else
	{
		emptyEncryptFileFields();
		emptyDecryptFileFields();
	}
}

void CCryptDialog::encryptFile()
{
	bool ok;
	QString password = QInputDialog::getText(this, tr("Password"),
										tr("Enter a password:"), QLineEdit::Password,
										"", &ok);
	if (ok)
	{
		if (!password.isEmpty())
		{
			QString targetFileName;
			targetFileName = QString(currentFileInfo.absoluteFilePath());
			targetFileName = targetFileName + ".cpt";
			QFileInfo targetFile;
			targetFile = QFileInfo(targetFileName);
			if (targetFile.isFile())
			{
				int reply = QMessageBox::warning(this, tr("Replace file confirmation"),
								tr("The target file already exists. Do you want to overwrite it ?"),
								tr("&Yes"),
								tr("&No"));
				if (reply == 1)
					return;
			}
			
				
			QString program = "ccrypt";
			QStringList arguments;
			arguments << "-e" << "-f" << "-K" << password << currentFileInfo.absoluteFilePath() ;
			
			QProcess *encryptProcess = new QProcess(this);
			encryptProcess->start(program, arguments);
			
			encryptProcess->waitForFinished();
			int returnCode = encryptProcess->exitCode ();

			switch (returnCode) {
				case 0 :
					QMessageBox::information(this, "Qccrypt", "Encryption successfull.", "&Ok");
					emptyEncryptFileFields();
					emptyDecryptFileFields();
					break;
				default :
					QMessageBox::critical(this, "Qccrypt", "Encryption error.", "&Ok");
			}
		}
		else
			QMessageBox::warning(this, "Qccrypt", "You must enter a password to encrypt this file !", "&Ok");
	}
}

void CCryptDialog::emptyEncryptFileFields()
{
	ccdlg_ui.encryptFileNameEdit->setText("");

	ccdlg_ui.encryptPathValueLabel->setText("");
	
	ccdlg_ui.encryptSizeValueLabel->setText("");
	
	ccdlg_ui.encryptLastReadValueLabel->setText("");

    ccdlg_ui.encryptLastModValueLabel->setText("");

	ccdlg_ui.encryptButton->setEnabled ( false );
}

void CCryptDialog::setDecryptOpenFileName()
{
	QString fileName = QFileDialog::getOpenFileName(this,
								tr("Select a file to decrypt"),
								ccdlg_ui.encryptFileNameEdit->text(),
								tr("ccrypt crypted file (*.cpt)"));
	
	if (!fileName.isEmpty())
	{
		currentFileInfo.setFile(fileName);

		if (currentFileInfo.exists())
		{
			currentFileInfo.setFile(fileName);
			ccdlg_ui.decryptFileNameEdit->setText(fileName);
		
			ccdlg_ui.decryptPathValueLabel->setText(currentFileInfo.absoluteFilePath());
			
			qlonglong size = currentFileInfo.size()/1024;
			if ((size == 0) and (currentFileInfo.size()) > 0 )
				size = 1;
			ccdlg_ui.decryptSizeValueLabel->setText(tr("%1 K").arg(size));
			
			ccdlg_ui.decryptLastReadValueLabel->setText(currentFileInfo.lastRead().toString());
	
		    ccdlg_ui.decryptLastModValueLabel->setText(currentFileInfo.lastModified().toString());
		    
	    	ccdlg_ui.decryptButton->setEnabled ( true );
   		}
   		else
   		{
   			emptyEncryptFileFields();
   			emptyDecryptFileFields();
  		}
	}
	else
	{
		emptyEncryptFileFields();
		emptyDecryptFileFields();
	}
}


void CCryptDialog::emptyDecryptFileFields()
{
	ccdlg_ui.decryptFileNameEdit->setText("");

	ccdlg_ui.decryptPathValueLabel->setText("");
	
	ccdlg_ui.decryptSizeValueLabel->setText("");
	
	ccdlg_ui.decryptLastReadValueLabel->setText("");

    ccdlg_ui.decryptLastModValueLabel->setText("");

	ccdlg_ui.decryptButton->setEnabled ( false );
}

void CCryptDialog::decryptFile()
{
	bool ok;
	QString password = QInputDialog::getText(this, tr("Password"),
										tr("Enter the password:"), QLineEdit::Password,
										"", &ok);
	if (ok)
	{
		if (!password.isEmpty())
		{
			QString targetFileName;
			targetFileName = QString(currentFileInfo.absoluteFilePath());
			targetFileName = targetFileName.left(targetFileName.length() - 4);
			QFileInfo targetFile;
			targetFile = QFileInfo(targetFileName);
			if (targetFile.isFile())
			{
				int reply = QMessageBox::warning(this, tr("Replace file confirmation"),
								tr("The target file already exists. Do you want to overwrite it ?"),
								tr("&Yes"),
								tr("&No"));
				if (reply == 1)
					return;
			}

			QString program = "ccrypt";
			QStringList arguments;
			arguments << "-d" << "-f" << "-K" << password << currentFileInfo.absoluteFilePath() ;
			
			QProcess *encryptProcess = new QProcess(this);
			encryptProcess->start(program, arguments);
			
			encryptProcess->waitForFinished();

			int returnCode = encryptProcess->exitCode ();

			switch (returnCode) {
				case 0 :
					QMessageBox::information(this, "Qccrypt", "Decryption successfull.", "&Ok");
					emptyEncryptFileFields();
					emptyDecryptFileFields();
					break;
				case 4 :
					QMessageBox::critical(this, "Qccrypt", "The password specified was incorrect !", "&Ok");
					break;
				default :
					QMessageBox::critical(this, "Qccrypt", "Decryption error.", "&Ok");
			}
		}
		else
			QMessageBox::warning(this, "Qccrypt", "You must enter the password to decrypt this file !", "&Ok");
	}
}

void CCryptDialog::about()
{
	QMessageBox::information(this, "About Qccrypt 0.1.2", "Qccrypt is an Open Source frontend for ccrypt encryption/decryption tool.\nIt is written in C++ with QT4 libraries by Philippe Beaureilles.\n\nThis program is provided under the terms of the GPL license 2.0 and later.\n\nhttp://qccrypt.free/", "&Ok");
}
